<?php
/**
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Webapi;

use Laminas\Code\Reflection\ClassReflection;
use Magento\CustomAttributeSerializable\Model\AttributesConfigurationPool;
use Magento\Framework\Api\AttributeValueFactory;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SimpleDataObjectConverter;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\SerializationException;
use Magento\Framework\ObjectManager\ConfigInterface;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Phrase;
use Magento\Framework\Reflection\MethodsMap;
use Magento\Framework\Reflection\TypeProcessor;
use Magento\Framework\Webapi\CustomAttributeTypeLocatorInterface;
use Magento\Framework\Webapi\ServiceTypeToEntityTypeMap;
use Magento\Framework\Webapi\Validator\IOLimit\DefaultPageSizeSetter;
use Magento\Framework\Webapi\Validator\ServiceInputValidatorInterface;

/**
 * @inheritDoc
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class ServiceInputProcessor extends \Magento\Framework\Webapi\ServiceInputProcessor
{
    /**
     * @var \Magento\Framework\Reflection\TypeProcessor
     */
    protected $typeProcessor;

    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    protected $objectManager;

    /**
     * @var \Magento\Framework\Api\AttributeValueFactory
     */
    protected $attributeValueFactory;

    /**
     * @var \Magento\Framework\Webapi\CustomAttributeTypeLocatorInterface
     */
    protected $customAttributeTypeLocator;

    /**
     * @var \Magento\Framework\Reflection\MethodsMap
     */
    protected $methodsMap;

    /**
     * @var \Magento\Framework\Reflection\NameFinder
     */
    private $nameFinder;

    /**
     * @var ConfigInterface
     */
    private $config;

    /**
     * @var ServiceInputValidatorInterface
     */
    private $serviceInputValidator;

    /**
     * @var int
     */
    private $defaultPageSize;

    /**
     * @var DefaultPageSizeSetter|null
     */
    private $defaultPageSizeSetter;

    /**
     * @var array
     */
    private $methodReflectionStorage = [];

    /**
     * Initialize dependencies.
     *
     * @param TypeProcessor $typeProcessor
     * @param ObjectManagerInterface $objectManager
     * @param AttributeValueFactory $attributeValueFactory
     * @param CustomAttributeTypeLocatorInterface $customAttributeTypeLocator
     * @param MethodsMap $methodsMap
     * @param ServiceTypeToEntityTypeMap|null $serviceTypeToEntityTypeMap
     * @param ConfigInterface|null $config
     * @param array $customAttributePreprocessors
     * @param ServiceInputValidatorInterface|null $serviceInputValidator
     * @param int $defaultPageSize
     * @param DefaultPageSizeSetter|null $defaultPageSizeSetter
     * @param AttributesConfigurationPool|null $attributesConfigurationPool
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        TypeProcessor $typeProcessor,
        ObjectManagerInterface $objectManager,
        AttributeValueFactory $attributeValueFactory,
        CustomAttributeTypeLocatorInterface $customAttributeTypeLocator,
        MethodsMap $methodsMap,
        ?ServiceTypeToEntityTypeMap $serviceTypeToEntityTypeMap = null,
        ?ConfigInterface $config = null,
        ?array $customAttributePreprocessors = [],
        ?ServiceInputValidatorInterface $serviceInputValidator = null,
        ?int $defaultPageSize = 20,
        ?DefaultPageSizeSetter $defaultPageSizeSetter = null,
        private ?AttributesConfigurationPool $attributesConfigurationPool = null
    ) {
        $this->typeProcessor = $typeProcessor;
        $this->objectManager = $objectManager;
        $this->attributeValueFactory = $attributeValueFactory;
        $this->customAttributeTypeLocator = $customAttributeTypeLocator;
        $this->methodsMap = $methodsMap;
        $serviceTypeToEntityTypeMap = $serviceTypeToEntityTypeMap
            ?: ObjectManager::getInstance()->get(ServiceTypeToEntityTypeMap::class);
        $this->config = $config
            ?: ObjectManager::getInstance()->get(ConfigInterface::class);
        $this->serviceInputValidator = $serviceInputValidator
            ?: ObjectManager::getInstance()->get(ServiceInputValidatorInterface::class);
        $this->defaultPageSize = $defaultPageSize >= 10 ? $defaultPageSize : 10;
        $this->defaultPageSizeSetter = $defaultPageSizeSetter ?? ObjectManager::getInstance()
            ->get(DefaultPageSizeSetter::class);
        $this->attributesConfigurationPool = $attributesConfigurationPool ?? ObjectManager::getInstance()
            ->get(AttributesConfigurationPool::class);

        parent::__construct(
            $typeProcessor,
            $objectManager,
            $attributeValueFactory,
            $customAttributeTypeLocator,
            $methodsMap,
            $serviceTypeToEntityTypeMap,
            $config,
            $customAttributePreprocessors,
            $serviceInputValidator,
            $defaultPageSize,
            $defaultPageSizeSetter
        );
    }

    /**
     * The getter function to get the new NameFinder dependency
     *
     * @return \Magento\Framework\Reflection\NameFinder
     *
     * @deprecated 100.1.0
     * @see nothing
     */
    private function getNameFinder()
    {
        if ($this->nameFinder === null) {
            $this->nameFinder = ObjectManager::getInstance()
                ->get(\Magento\Framework\Reflection\NameFinder::class);
        }
        return $this->nameFinder;
    }

    /**
     * Retrieve constructor data
     *
     * @param string $className
     * @param array $data
     * @return array
     * @throws \ReflectionException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function getConstructorData(string $className, array $data): array
    {
        $preferenceClass = $this->config->getPreference($className);
        $class = new ClassReflection($preferenceClass ?: $className);

        try {
            $constructor = $class->getMethod('__construct');
        } catch (\ReflectionException $e) {
            $constructor = null;
        }

        if ($constructor === null) {
            return [];
        }

        $res = [];
        $parameters = $constructor->getParameters();
        foreach ($parameters as $parameter) {
            if (isset($data[$parameter->getName()])) {
                $parameterType = $this->typeProcessor->getParamType($parameter);

                // Allow only simple types or Api Data Objects
                if (!($this->typeProcessor->isTypeSimple($parameterType)
                   || preg_match('~\\\\?\w+\\\\\w+\\\\Api\\\\Data\\\\~', $parameterType) === 1
                )) {
                    continue;
                }

                try {
                    $res[$parameter->getName()] = $this->convertValue($data[$parameter->getName()], $parameterType);
                } catch (\ReflectionException $e) {
                    // Parameter was not correclty declared or the class is uknown.
                    // By not returing the contructor value, we will automatically fall back to the "setters" way.
                    continue;
                }
            }
        }

        return $res;
    }

    /**
     * Creates a new instance of the given class and populates it with the array of data. The data can
     * be in different forms depending on the adapter being used, REST vs. SOAP. For REST, the data is
     * in snake_case (e.g. tax_class_id) while for SOAP the data is in camelCase (e.g. taxClassId).
     *
     * @param string $className
     * @param array $data
     * @return object the newly created and populated object
     * @throws \Exception
     * @throws SerializationException
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    protected function _createFromArray($className, $data)
    {
        $data = is_array($data) ? $data : [];
        // convert to string directly to avoid situations when $className is object
        // which implements __toString method like \ReflectionObject
        $className = (string) $className;
        if (is_subclass_of($className, \SimpleXMLElement::class)
            || is_subclass_of($className, \DOMElement::class)) {
            throw new SerializationException(
                new Phrase('Invalid data type')
            );
        }
        $class = new ClassReflection($className);
        if (is_subclass_of($className, self::EXTENSION_ATTRIBUTES_TYPE)) {
            $className = substr($className, 0, -strlen('Interface'));
        }

        // Primary method: assign to constructor parameters
        $constructorArgs = $this->getConstructorData($className, $data);
        $object = $this->objectManager->create($className, $constructorArgs);

        // Secondary method: fallback to setter methods
        foreach ($data as $propertyName => $value) {
            if (isset($constructorArgs[$propertyName])) {
                continue;
            }

            // Converts snake_case to uppercase CamelCase to help form getter/setter method names
            // This use case is for REST only. SOAP request data is already camel cased
            $camelCaseProperty = SimpleDataObjectConverter::snakeCaseToUpperCamelCase($propertyName);

            // As interfaces of entities with serializable custom attributes do not have setCustomAttributes method
            // as they don't extend \Magento\Framework\Api\CustomAttributesDataInterface
            // we need to check if the method exists in the class and call it directly.
            if ($camelCaseProperty === 'CustomAttributes' &&
                in_array(trim($className, '\\'), $this->attributesConfigurationPool->getApiInterfaces()) &&
                method_exists($object, 'setCustomAttributes')
            ) {
                $object->setCustomAttributes($this->convertCustomAttributeValue($value, $className));
                continue;
            }

            try {
                $methodName = $this->getNameFinder()->getGetterMethodName($class, $camelCaseProperty);
                if (!isset($this->methodReflectionStorage[$className . $methodName])) {
                    $this->methodReflectionStorage[$className . $methodName] = $class->getMethod($methodName);
                }
                $methodReflection = $this->methodReflectionStorage[$className . $methodName];
                if ($methodReflection->isPublic()) {
                    $returnType = $this->typeProcessor->getGetterReturnType($methodReflection)['type'];
                    try {
                        $setterName = $this->getNameFinder()->getSetterMethodName($class, $camelCaseProperty);
                    } catch (\Exception $e) {
                        if (empty($value)) {
                            continue;
                        } else {
                            throw $e;
                        }
                    }
                    try {
                        if ($camelCaseProperty === 'CustomAttributes') {
                            $setterValue = $this->convertCustomAttributeValue($value, $className);
                        } else {
                            $setterValue = $this->convertValue($value, $returnType);
                        }
                    } catch (SerializationException $e) {
                        throw new SerializationException(
                            new Phrase(
                                'Error occurred during "%field_name" processing. %details',
                                ['field_name' => $propertyName, 'details' => $e->getMessage()]
                            )
                        );
                    }
                    if (is_string($setterValue) && $this->validateParamsValue($setterValue)) {
                        throw new InputException(
                            new Phrase(
                                '"%field_name" does not contains valid value.',
                                ['field_name' => $propertyName]
                            )
                        );
                    }
                    $this->serviceInputValidator->validateEntityValue($object, $propertyName, $setterValue);
                    $object->{$setterName}($setterValue);
                }
            } catch (\LogicException $e) {
                $this->processInputErrorForNestedSet([$camelCaseProperty]);
            }
        }

        if ($object instanceof SearchCriteriaInterface) {
            $this->defaultPageSizeSetter->processSearchCriteria($object, $this->defaultPageSize);
        }

        return $object;
    }

    /**
     * Validate input param value
     *
     * @param string $value
     * @return bool
     */
    private function validateParamsValue(string $value)
    {
        return preg_match('/<script\b[^>]*>(.*?)<\/script>/is', $value);
    }

    /**
     * Process an input error for child parameters
     *
     * @param array $inputError
     * @return void
     * @throws InputException
     */
    private function processInputErrorForNestedSet(array $inputError): void
    {
        if (!empty($inputError)) {
            $exception = new InputException();
            foreach ($inputError as $errorParamField) {
                $exception->addError(
                    new Phrase(
                        '"%fieldName" is not supported. Correct the field name and try again.',
                        ['fieldName' => $errorParamField]
                    )
                );
            }
            if ($exception->wasErrorAdded()) {
                throw $exception;
            }
        }
    }
}
