<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\CustomAttributeSerializable\Test\Api\Quote;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\EmailNotConfirmedException;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Integration\Api\CustomerTokenServiceInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\Quote;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\ObjectManager;
use Magento\TestFramework\TestCase\WebapiAbstract;

/**
 * Tests cart custom attributes
 */
class CartCustomAttributesTest extends WebapiAbstract
{
    use CartHelper;

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * @inheritDoc
     */
    protected function tearDown(): void
    {
        try {
            /** @var CartRepositoryInterface $quoteRepository */
            $quoteRepository = $this->objectManager->get(CartRepositoryInterface::class);
            $cart = $this->getCart('test01');
            $quoteRepository->delete($cart);
        } catch (\InvalidArgumentException $e) {
            // Do nothing if cart fixture was not used
        }
        parent::tearDown();
    }

    /**
     * Tests setting custom attributes via POST endpoint /V1/carts/:cartId/customAttributes
     *
     * @magentoApiDataFixture Magento/Sales/_files/quote.php
     */
    public function testSetCartCustomAttributesViaPost()
    {
        $this->_markTestAsRestOnly();
        $cart = $this->getCart('test01');
        $cartId = $cart->getId();

        $customAttributes = [
            [
                'attribute_code' => 'gift_wrap_requested',
                'value' => true
            ],
            [
                'attribute_code' => 'delivery_instructions',
                'value' => 'Leave at front door'
            ],
            [
                'attribute_code' => 'shipping_preferences',
                'value' => [
                    'primary' => [
                        'address' => [
                            'street' => '123 Main St',
                            'city' => 'New York',
                            'state' => 'NY'
                        ],
                        'options' => [
                            'signature_required' => true,
                        ]
                    ]
                ]
            ]
        ];

        $serviceInfo = [
            'rest' => [
                'resourcePath' => '/V1/carts/' . $cartId . '/customAttributes',
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $requestData = [
            'customAttributes' => $customAttributes
        ];

        $response = $this->_webApiCall($serviceInfo, $requestData);
        self::assertTrue($response, 'Expected true response from custom attributes save');

        $getServiceInfo = [
            'rest' => [
                'resourcePath' => '/V1/carts/' . $cartId,
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $cartData = $this->_webApiCall($getServiceInfo, ['cartId' => $cartId]);

        self::assertArrayHasKey('custom_attributes', $cartData);
        self::assertCount(3, $cartData['custom_attributes']);

        $actualAttributes = [];
        foreach ($cartData['custom_attributes'] as $attr) {
            $actualAttributes[$attr['attribute_code']] = $attr['value'];
        }

        self::assertArrayHasKey('gift_wrap_requested', $actualAttributes);
        self::assertArrayHasKey('delivery_instructions', $actualAttributes);
        self::assertArrayHasKey('shipping_preferences', $actualAttributes);
        self::assertTrue($actualAttributes['gift_wrap_requested']);
        self::assertEquals('Leave at front door', $actualAttributes['delivery_instructions']);

        $shippingPrefs = $actualAttributes['shipping_preferences'];
        self::assertEquals('123 Main St', $shippingPrefs['primary']['address']['street'] ?? '');
        self::assertEquals('New York', $shippingPrefs['primary']['address']['city'] ?? '');
        self::assertTrue($shippingPrefs['primary']['options']['signature_required'] ?? false);
    }

    /**
     * Tests cart are fetched with custom attributes.
     *
     * @magentoApiDataFixture Magento/Sales/_files/quote.php
     */
    public function testGetCartWithCustomAttributes()
    {
        $this->_markTestAsRestOnly();
        $cart = $this->getCart('test01');
        $cartId = $cart->getId();
        $cart->setCustomAttribute('attr_one', 'value_one');
        $cart->setCustomAttribute('attr_two', 'value_two');
        $quoteRepository = $this->objectManager->get(CartRepositoryInterface::class);
        $quoteRepository->save($cart);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => '/V1/carts/' . $cartId,
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $requestData = ['cartId' => $cartId];
        $cartData = $this->_webApiCall($serviceInfo, $requestData);
        self::assertEquals($cart->getId(), $cartData['id']);
        self::assertEquals(
            [
                [
                    'attribute_code' => 'attr_one',
                    'value' => 'value_one'
                ],
                [
                    'attribute_code' => 'attr_two',
                    'value' => 'value_two'
                ],
            ],
            $cartData['custom_attributes']
        );
    }

    /**
     * Tests that cart custom attributes are saved.
     *
     * @magentoApiDataFixture Magento/Checkout/_files/quote_with_shipping_method.php
     */
    public function testUpdateCartCustomAttributes()
    {
        $this->_markTestAsRestOnly();
        $token = $this->getToken();

        /** @var Quote $quote */
        $quote = $this->getCart('test_order_1');

        $requestData = $this->getRequestData($quote->getId());

        $serviceInfo = [
            'rest' => [
                'resourcePath' => '/V1/carts/mine',
                'httpMethod'   => Request::HTTP_METHOD_PUT,
                'token'        => $token
            ],
        ];

        $this->_webApiCall($serviceInfo, $requestData);

        $quote->loadActive($requestData['quote']['id']);

        self::assertEquals(
            [
                'attr_one' => 'value_one',
                'attr_two' => 'value_two'
            ],
            $this->customAttributeConverter->toSerializableFormat($quote->getCustomAttributes()),
        );
    }

    /**
     * Request's data with custom attributes
     *
     * @param int $quoteId
     * @return array
     */
    private function getRequestData(int $quoteId): array
    {
        $requestData['quote'] = [
            'id' => $quoteId,
            'store_id' => 1,
            'customer' => [
                'id' => 1,
            ],
            'custom_attributes' => [
                [
                    'attribute_code' => 'attr_one',
                    'value' => 'value_one'
                ],
                [
                    'attribute_code' => 'attr_two',
                    'value' => 'value_two'
                ],
            ]
        ];

        return $requestData;
    }

    /**
     * Request to api for the current user token.
     *
     * @return string
     * @throws AuthenticationException
     * @throws EmailNotConfirmedException
     */
    private function getToken(): string
    {
        $customerTokenService = $this->objectManager->create(
            CustomerTokenServiceInterface::class
        );

        return $customerTokenService->createCustomerAccessToken('customer@example.com', 'password');
    }
}
