<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Model;

use Exception;
use Magento\CustomAttributeSerializable\Api\CustomAttributesManagementInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Framework\Api\AttributeInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotSaveException;
use Psr\Log\LoggerInterface;

/**
 * @inheritDoc
 */
class CustomAttributesManagement implements CustomAttributesManagementInterface
{
    /**
     * @param CartRepositoryInterface $cartRepository
     * @param LoggerInterface $logger
     */
    public function __construct(
        private readonly CartRepositoryInterface $cartRepository,
        private readonly LoggerInterface $logger
    ) {
    }

    /**
     * Save custom attributes for quote by quote ID
     *
     * @param int $cartId
     * @param AttributeInterface[] $customAttributes
     * @return bool
     * @throws NoSuchEntityException
     * @throws CouldNotSaveException
     */
    public function saveByQuoteId(int $cartId, array $customAttributes): bool
    {
        try {
            $quote = $this->cartRepository->get($cartId);

            if (!$quote->getId()) {
                throw new NoSuchEntityException(__('Quote with ID "%1" does not exist.', $cartId));
            }

            $quote->setCustomAttributes($customAttributes);
            $this->cartRepository->save($quote);

            return true;
        } catch (NoSuchEntityException $e) {
            throw $e;
        } catch (Exception $e) {
            $errorMessage = __('Could not save custom attributes for quote ID "%1": %2', $cartId, $e->getMessage());
            $this->logger->error($errorMessage);
            throw new CouldNotSaveException($errorMessage, $e);
        }
    }
}
