<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Webapi;

use Magento\CustomAttributeSerializable\Model\AttributesConfigurationPool;
use Magento\Framework\Api\AttributeInterface;
use Magento\Framework\Api\AttributeValue;
use Magento\Framework\Api\ExtensibleDataInterface;
use Magento\Framework\Api\ExtensionAttribute\JoinProcessorInterface;
use Magento\Framework\Api\ExtensionAttributesFactory;
use Magento\Framework\Api\ObjectFactory;
use Magento\Framework\Reflection\DataObjectProcessor;
use Magento\Framework\Reflection\MethodsMap;
use Magento\Framework\Reflection\TypeProcessor;

/**
 * Override DataObjectHelper to set custom attributes as complex type
 */
class DataObjectHelper extends \Magento\Framework\Api\DataObjectHelper
{
    /**
     * @param ObjectFactory $objectFactory
     * @param DataObjectProcessor $objectProcessor
     * @param TypeProcessor $typeProcessor
     * @param ExtensionAttributesFactory $extensionFactory
     * @param JoinProcessorInterface $joinProcessor
     * @param MethodsMap $methodsMapProcessor
     * @param AttributesConfigurationPool $attributesConfigurationPool
     */
    public function __construct(
        ObjectFactory $objectFactory,
        \Magento\Framework\Reflection\DataObjectProcessor $objectProcessor,
        \Magento\Framework\Reflection\TypeProcessor $typeProcessor,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\ExtensionAttribute\JoinProcessorInterface $joinProcessor,
        MethodsMap $methodsMapProcessor,
        private AttributesConfigurationPool $attributesConfigurationPool
    ) {
        parent::__construct(
            $objectFactory,
            $objectProcessor,
            $typeProcessor,
            $extensionFactory,
            $joinProcessor,
            $methodsMapProcessor
        );
    }

    /**
     * Set custom attributes for entities with custom attributes serialization
     *
     * @param mixed $dataObject
     * @param string $getterMethodName
     * @param string $methodName
     * @param array $value
     * @param string $interfaceName
     * @return $this
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     */
    protected function setComplexValue(
        $dataObject,
        $getterMethodName,
        $methodName,
        array $value,
        $interfaceName
    ) {
        if (in_array($methodName, ['setCustomAttributes', 'setCustomAttributesSerializable']) &&
            in_array(trim($interfaceName, '\\'), $this->attributesConfigurationPool->getApiInterfaces()) &&
            method_exists($dataObject, $methodName)
        ) {
            $dataObject->{$methodName}($value);
            return $this;
        }

        return parent::setComplexValue($dataObject, $getterMethodName, $methodName, $value, $interfaceName);
    }

    /**
     * Set custom attributes using the $attributeKeys parameter.
     *
     * @param mixed $dataObject
     * @param array $data
     * @param array $attributeKeys
     * @return array
     */
    public function setCustomAttributes(mixed $dataObject, array $data, array $attributeKeys): array
    {
        foreach ($attributeKeys as $attributeKey) {
            if ($dataObject instanceof ExtensibleDataInterface
                && !empty($data[$attributeKey])
            ) {
                foreach ($data[$attributeKey] as $customAttribute) {
                    if ($customAttribute instanceof AttributeValue) {
                        $dataObject->setCustomAttribute(
                            $customAttribute->getAttributeCode(),
                            $customAttribute->getValue()
                        );
                    } else {
                        $dataObject->setCustomAttribute(
                            $customAttribute[AttributeInterface::ATTRIBUTE_CODE],
                            $customAttribute[AttributeInterface::VALUE]
                        );
                    }
                }
                unset($data[$attributeKey]);
            }
        }
        return $data;
    }
}
