<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Controller\Adminhtml\Order;

use InvalidArgumentException;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\CustomAttributeSerializable\Model\CustomAttributes\RequestCustomAttributeConverter;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Throwable;

/**
 * Controller to handle saving custom attributes for an order.
 */
class Save extends Action implements HttpPostActionInterface
{
    public const ADMIN_RESOURCE = 'Magento_Sales::actions_edit';

    /**
     * @param Context $context
     * @param OrderRepositoryInterface $orderRepository
     * @param JsonFactory $resultJsonFactory
     * @param RequestCustomAttributeConverter $attributeConverter
     */
    public function __construct(
        Context $context,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly JsonFactory $resultJsonFactory,
        private readonly RequestCustomAttributeConverter $attributeConverter,
    ) {
        parent::__construct($context);
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $result = $this->resultJsonFactory->create();

        try {
            $orderId = (int)$this->getRequest()->getParam('order_id');
            if (!$orderId) {
                throw new LocalizedException(__('Missing order_id.'));
            }

            $attributes = $this->getRequest()->getParam('custom_order_attributes', []);

            if (!is_array($attributes)) {
                throw new LocalizedException(__('Custom attributes has a wrong format.'));
            }

            $customAttributes = $this->attributeConverter->convert($attributes);

            /** @var OrderInterface $order */
            $order = $this->orderRepository->get($orderId);
            $order->setCustomAttributes($customAttributes);
            $this->orderRepository->save($order);

            $this->messageManager->addSuccessMessage(
                __('Custom attributes have saved for order: "%1".', $order->getIncrementId())
            );

            return $result->setData([
                'success' => true,
                'message' => __('Custom attributes saved.'),
            ]);
        } catch (Throwable $e) {
            return $result->setHttpResponseCode(400)->setData([
                'success' => false,
                'message' => $e instanceof LocalizedException ? $e->getMessage() : __('Error saving attributes.'),
            ]);
        }
    }
}
