<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Test\Api\Order;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\ObjectManager;
use Magento\TestFramework\TestCase\WebapiAbstract;
use Magento\Sales\Model\Order;
use Magento\Framework\Webapi\Rest\Request;

/**
 * Tests order custom attributes
 */
class OrderCustomAttributesTest extends WebapiAbstract
{
    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * Tests that order custom attributes are returned via API
     *
     * @magentoApiDataFixture Magento/Sales/_files/order.php
     */
    public function testOrderGetCustomAttributes()
    {
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');
        $order->setCustomAttribute('attr_one', 'value_one');
        $order->setCustomAttribute('attr_two', 'value_two');
        $orderRepository = $this->objectManager->get(OrderRepositoryInterface::class);
        $orderRepository->save($order);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => '/V1/orders/' . $order->getId(),
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $response = $this->_webApiCall($serviceInfo);

        self::assertEquals(
            [
                [
                    'attribute_code' => 'attr_one',
                    'value' => 'value_one'
                ],
                [
                    'attribute_code' => 'attr_two',
                    'value' => 'value_two'
                ],
            ],
            $response['custom_attributes']
        );
    }

    /**
     * Tests that order custom attributes are updated via API
     *
     * @magentoApiDataFixture Magento/Sales/_files/order.php
     */
    public function testOrderCustomAttributesUpdate()
    {
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        $serviceInfo = [
            'rest' => [
                'resourcePath' => '/V1/orders',
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $requestData = [
            'entity' => [
                'entity_id' => $order->getId(),
                'custom_attributes' => [
                    [
                        'attribute_code' => 'attr_one',
                        'value' => 'value_one'
                    ],
                    [
                        'attribute_code' => 'attr_two',
                        'value' => 'value_two'
                    ],
                ]
            ],
        ];

        $this->_webApiCall($serviceInfo, $requestData);

        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        self::assertEquals(
            [
                'attr_one' => 'value_one',
                'attr_two' => 'value_two'
            ],
            $this->customAttributeConverter->toSerializableFormat($order->getCustomAttributes()),
        );

        $requestData = [
            'entity' => [
                'entity_id' => $order->getId(),
                'custom_attributes' => [
                    [
                        'attribute_code' => 'attr_one',
                        'value' => 'value_one'
                    ],
                    [
                        'attribute_code' => 'attr_two',
                        'value' => 'value_two'
                    ],
                    [
                        'attribute_code' => 'attr_three',
                        'value' => 'value_three'
                    ],
                ]
            ],
        ];

        $this->_webApiCall($serviceInfo, $requestData);

        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        self::assertEquals(
            [
                'attr_one' => 'value_one',
                'attr_two' => 'value_two',
                'attr_three' => 'value_three',
            ],
            $this->customAttributeConverter->toSerializableFormat($order->getCustomAttributes()),
        );
    }
}
