<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Test\Api\Invoice;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Sales\Api\InvoiceRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Invoice;
use Magento\Sales\Model\ResourceModel\Order\Invoice\Collection;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\WebapiAbstract;

/**
 * Tests invoice custom attributes
 */
class InvoiceCustomAttributesTest extends WebapiAbstract
{
    const RESOURCE_PATH = '/V1/invoices';

    /**
     * @var ObjectManagerInterface
     */
    protected $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * Checks that invoice and invoice item custom attributes are returned in the response
     *
     * @magentoApiDataFixture Magento/Sales/_files/invoice.php
     */
    public function testGetInvoiceCustomAttributes()
    {
        /** @var Invoice $invoice */
        $invoiceCollection = $this->objectManager->get(Collection::class);
        $invoice = $invoiceCollection->getFirstItem();
        $invoice->setCustomAttribute('attr_one', 'value_one');
        $invoice->setCustomAttribute('attr_two', 'value_two');
        $invoiceRepository = $this->objectManager->get(InvoiceRepositoryInterface::class);

        foreach ($invoice->getItems() as $invoiceItem) {
            $invoiceItem->setCustomAttribute('attr_one', 'value_one');
            $invoiceItem->setCustomAttribute('attr_two', 'value_two');
        }
        $invoiceRepository->save($invoice);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH . '/' . $invoice->getId(),
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];
        $invoiceResponseData = $this->_webApiCall($serviceInfo, ['id' => $invoice->getId()]);
        self::assertEquals($this->getCustomAttributes(), $invoiceResponseData['custom_attributes'] ?? []);
        self::assertArrayHasKey('items', $invoiceResponseData);
        foreach ($invoiceResponseData['items'] as $item) {
            self::assertEquals($this->getCustomAttributes(), $item['custom_attributes'] ?? []);
        }
    }

    /**
     * Tests that invoice and invoice item custom attributes are saved via rest API
     *
     * @magentoApiDataFixture Magento/Sales/_files/order.php
     */
    public function testCreatAndUpdateInvoiceWithCustomAttribute()
    {
        /** @var Order $order */
        $order = $this->objectManager->create(Order::class)->loadByIncrementId('100000001');
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH,
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $orderItems = $order->getAllItems();
        $data = [
            'order_id' => $order->getId(),
            'custom_attributes' => $this->getCustomAttributes(),
            'items' => [
                [
                    'orderItemId' => $orderItems[0]->getId(),
                    'qty' => 2,
                    'sku' => 'sku' . uniqid(),
                    'custom_attributes' => $this->getCustomAttributes()
                ],
            ],
        ];
        $response = $this->_webApiCall($serviceInfo, ['entity' => $data]);
        self::assertArrayHasKey('entity_id', $response);
        self::assertArrayHasKey('custom_attributes', $response);

        $this->assertInvoiceAttributes(
            $response['entity_id'],
            [
                'attr_one' => 'value_one',
                'attr_two' => 'value_two'
            ]
        );

        $data['entity_id'] = $response['entity_id'];
        $data['custom_attributes'] = $this->getCustomAttributesUpdated();
        $data['items'][0]['custom_attributes'] = $this->getCustomAttributesUpdated();
        $data['items'][0]['entity_id'] = $response['items'][0]['entity_id'];
        $data['items'][0]['parent_id'] = $response['items'][0]['parent_id'];

        $updateResponse = $this->_webApiCall($serviceInfo, ['entity' => $data]);
        self::assertArrayHasKey('entity_id', $updateResponse);
        self::assertArrayHasKey('custom_attributes', $updateResponse);

        $this->assertInvoiceAttributes(
            $updateResponse['entity_id'],
            [
                'attr_one' => 'value_one_updated',
                'attr_two' => 'value_two_updated',
                'attr_three' => 'value_three',
            ]
        );
    }

    /**
     * Asserts that invoice and invoice item custom attributes are saved
     *
     * @param int $invoiceId
     * @param array $expectedAttributes
     * @return void
     */
    private function assertInvoiceAttributes(int $invoiceId, array $expectedAttributes): void
    {
        $invoiceRepository = $this->objectManager->create(InvoiceRepositoryInterface::class);
        $invoice = $invoiceRepository->get($invoiceId);
        self::assertEquals(
            $expectedAttributes,
            $this->customAttributeConverter->toSerializableFormat($invoice->getCustomAttributes())
        );
        foreach ($invoice->getItems() as $invoiceItem) {
            self::assertEquals(
                $expectedAttributes,
                $this->customAttributeConverter->toSerializableFormat($invoiceItem->getCustomAttributes())
            );
        }
    }

    /**
     * Returns custom attributes
     *
     * @return array
     */
    private function getCustomAttributes(): array
    {
        return [
            [
                'attribute_code' => 'attr_one',
                'value' => 'value_one'
            ],
            [
                'attribute_code' => 'attr_two',
                'value' => 'value_two'
            ],
        ];
    }

    /**
     * Returns custom attributes
     *
     * @return array
     */
    private function getCustomAttributesUpdated(): array
    {
        return [
            [
                'attribute_code' => 'attr_one',
                'value' => 'value_one_updated'
            ],
            [
                'attribute_code' => 'attr_two',
                'value' => 'value_two_updated'
            ],
            [
                'attribute_code' => 'attr_three',
                'value' => 'value_three'
            ],
        ];
    }
}
