<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Model;

use Magento\Framework\Api\AbstractExtensibleObject;
use Magento\Framework\Api\AttributeInterface;
use Magento\Framework\Api\AttributeValue;
use Magento\Framework\Api\CustomAttributesDataInterface;
use Magento\Framework\Model\AbstractExtensibleModel;

/**
 * Overrides the methods of model class to add manage custom attributes from the serialized field
 */
trait SerializableAttributesTrait
{
    /**
     * @var array List of attributes that should be ignored
     */
    private array $ignoreAttributes = [
        AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE,
        'address_type',
        'customer_address_id',
        'gift_cards',
    ];

    /**
     * @inheritDoc
     */
    public function getCustomAttributesCodes(): array
    {
        if (empty($this->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE)) ||
            !is_array($this->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE))
        ) {
            return [];
        }

        return array_keys($this->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE));
    }

    /**
     * Overwrite the method to add custom attributes.
     *
     * Skips checking if the attributes exists in the result of getCustomAttributesCodes method
     *
     * @param string $attributeCode
     * @param mixed $attributeValue
     * @return $this
     */
    public function setCustomAttribute($attributeCode, $attributeValue)
    {
        if (in_array($attributeCode, $this->getIgnoredAttributes())) {
            return $this;
        }

        $attribute = $this->createAttribute();
        $attribute->setAttributeCode($attributeCode)
            ->setValue($attributeValue);
        $this->_data[self::CUSTOM_ATTRIBUTES][$attributeCode] = $attribute;

        return $this;
    }

    /**
     * Overwrite the method to set array custom attributes.
     *
     * Skip checking if the attributes exists in the result of getCustomAttributesCodes method
     *
     * @param array $attributes
     * @return array
     */
    public function setCustomAttributes(array $attributes)
    {
        $data = [];
        foreach ($attributes as $code => $value) {
            if (!$value instanceof AttributeInterface) {
                $data[$code] = $this->createAttribute()
                    ->setAttributeCode($code)
                    ->setValue($value);
            } else {
                $data[$value->getAttributeCode()] = $value;
            }
        }

        return $this->_data[self::CUSTOM_ATTRIBUTES] = $data;
    }

    /**
     * Initialize custom attributes from serializable data
     */
    protected function initializeCustomAttributes()
    {
        parent::initializeCustomAttributes();

        foreach ($this->getCustomAttributesSerializable() as $code => $value) {
            if (!isset($this->_data[self::CUSTOM_ATTRIBUTES][$code])) {
                $this->setCustomAttribute($code, $value);
            }
        }
    }

    /**
     * Sets custom attributes serializable data
     *
     * @param array|string $customAttributes
     * @return CustomAttributesDataInterface
     */
    public function setCustomAttributesSerializable($customAttributes): CustomAttributesDataInterface
    {
        $this->setData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE, $customAttributes);

        return $this;
    }

    /**
     * Returns custom attributes serializable data
     *
     * @return array
     */
    public function getCustomAttributesSerializable(): array
    {
        $attributes = $this->getData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE);
        if (is_string($attributes)) {
            $attributes = json_decode($attributes, true);
        }

        return is_array($attributes) ? $attributes : [];
    }

    /**
     * Get list of attributes that should be ignored
     *
     * @return array
     */
    public function getIgnoredAttributes(): array
    {
        return $this->ignoreAttributes;
    }

    /**
     * Create attribute value object
     *
     * @return AttributeValue
     */
    private function createAttribute(): AttributeValue
    {
        if ($this instanceof AbstractExtensibleModel) {
            return $this->customAttributeFactory->create();
        } elseif ($this instanceof AbstractExtensibleObject) {
            return $this->attributeValueFactory->create();
        }

        return new AttributeValue();
    }
}
