<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2025 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\CustomAttributeSerializable\Test\Api\Tax;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Tax\Api\TaxClassRepositoryInterface;
use Magento\Tax\Model\ClassModel;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\ObjectManager;
use Magento\TestFramework\TestCase\WebapiAbstract;

/**
 * Tests for tax class custom attributes
 */
class TaxClassCustomAttributesTest extends WebapiAbstract
{
    const RESOURCE_PATH = '/V1/taxClasses';

    private const CODE_ONE = 'attr_one';
    private const CODE_TWO = 'attr_two';
    private const CODE_THREE = 'attr_three';
    private const VALUE_ONE = 'value_one';
    private const VALUE_ONE_UPDATED = 'value_one_updated';
    private const VALUE_TWO = 'value_two';
    private const VALUE_TWO_UPDATED = 'value_two_updated';
    private const VALUE_THREE = 'value_three';

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * Tests tax class custom attributes are contained in a response to a GET request
     *
     * @magentoApiDataFixture Magento/Tax/_files/tax_classes.php
     */
    public function testGetTaxClassWithCustomAttributes()
    {
        /** @var ClassModel $taxClass */
        $taxClass = $this->objectManager->create(
            ClassModel::class
        )->getCollection()->setClassTypeFilter(
            ClassModel::TAX_CLASS_TYPE_PRODUCT
        )->getFirstItem();

        $taxClass->setCustomAttribute(self::CODE_ONE, self::VALUE_ONE);
        $taxClass->setCustomAttribute(self::CODE_TWO, self::VALUE_TWO);

        $taxClassRepository = $this->objectManager->get(TaxClassRepositoryInterface::class);
        $taxClassRepository->save($taxClass);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH . '/' . $taxClass->getId(),
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $taxClassResponseData = $this->_webApiCall($serviceInfo, ['id' => $taxClass->getId()]);
        self::assertEquals($this->getCustomAttributes(), $taxClassResponseData['custom_attributes'] ?? []);
    }

    /**
     * Tests that tax class custom attributes are saved via REST API
     */
    public function testCreateAndUpdateCustomAttributesOnTaxClass()
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH,
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $data = [
            'class_name' => 'Test Tax Class',
            'class_type' => ClassModel::TAX_CLASS_TYPE_PRODUCT,
            'custom_attributes' => $this->getCustomAttributes(),
        ];
        $response = $this->_webApiCall($serviceInfo, ['taxClass' => $data]);
        self::assertNotEmpty($response);
        $taxClassId = (int) $response;
        $this->assertTaxClassAttributes(
            $taxClassId,
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_TWO => self::VALUE_TWO,
            ]
        );

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH . '/' . $taxClassId,
                'httpMethod' => Request::HTTP_METHOD_PUT,
            ],
        ];

        $data['custom_attributes'] = $this->getCustomAttributesUpdated();
        $updateResponse = $this->_webApiCall($serviceInfo, ['taxClass' => $data]);
        self::assertNotEmpty($updateResponse);
        $this->assertTaxClassAttributes(
            $taxClassId,
            [
                self::CODE_ONE => self::VALUE_ONE_UPDATED,
                self::CODE_TWO => self::VALUE_TWO_UPDATED,
                self::CODE_THREE => self::VALUE_THREE,
            ]
        );
    }

    /**
     * Asserts that tax class custom attributes are saved
     *
     * @param int $taxClassId
     * @param array $expectedAttributes
     * @return void
     */
    private function assertTaxClassAttributes(int $taxClassId, array $expectedAttributes): void
    {
        $taxClass = $this->objectManager->create(
            ClassModel::class
        )->load($taxClassId);

        self::assertEquals(
            $expectedAttributes,
            $this->customAttributeConverter->toSerializableFormat($taxClass->getCustomAttributes())
        );
    }

    /**
     * Returns custom attributes
     *
     * @return array
     */
    private function getCustomAttributes(): array
    {
        return [
            [
                'attribute_code' => self::CODE_ONE,
                'value' => self::VALUE_ONE
            ],
            [
                'attribute_code' => self::CODE_TWO,
                'value' => self::VALUE_TWO
            ],
        ];
    }

    /**
     * Returns a set of custom attributes with updated values
     *
     * @return array
     */
    private function getCustomAttributesUpdated(): array
    {
        return [
            [
                'attribute_code' => self::CODE_ONE,
                'value' => self::VALUE_ONE_UPDATED
            ],
            [
                'attribute_code' => self::CODE_TWO,
                'value' => self::VALUE_TWO_UPDATED
            ],
            [
                'attribute_code' => self::CODE_THREE,
                'value' => self::VALUE_THREE
            ]
        ];
    }
}
