<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\CustomAttributeSerializable\Test\Api\Quote;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product;
use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartItemInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\WebapiAbstract;

/**
 * Tests cart item custom attributes
 */
class CartItemCustomAttributesTest extends WebapiAbstract
{
    use CartHelper;

    const RESOURCE_PATH = '/V1/carts/';

    /**
     * @var ObjectManagerInterface
     */
    protected $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * Tests that cart items are returned with custom attributes
     *
     * @magentoApiDataFixture Magento/Checkout/_files/quote_with_items_and_custom_options_saved.php
     */
    public function testGetCartItemWithCustomAttributes()
    {
        $quote = $this->getCart('test_order_item_with_items_and_custom_options');

        /** @var  CartItemInterface $item */
        foreach ($quote->getAllItems() as $item) {
            $item->setCustomAttribute('attr_one', 'value_one');
            $item->setCustomAttribute('attr_two', 'value_two');
        }
        $quoteRepository = $this->objectManager->get(CartRepositoryInterface::class);
        $quoteRepository->save($quote);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH . $quote->getId() . '/items',
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $responseData = $this->_webApiCall($serviceInfo);
        foreach ($responseData as $itemData) {
            self::assertEquals(
                [
                    [
                        'attribute_code' => 'attr_one',
                        'value' => 'value_one'
                    ],
                    [
                        'attribute_code' => 'attr_two',
                        'value' => 'value_two'
                    ],
                ],
                $itemData['custom_attributes'] ?? []
            );
        }
    }

    /**
     * Tests that cart item custom attributes are saved.
     *
     * @magentoApiDataFixture Magento/Checkout/_files/quote_with_address_saved.php
     * @magentoApiDataFixture Magento/Catalog/_files/product_without_options.php
     */
    public function testAddItemWithCustomAttributes()
    {
        /** @var  Product $product */
        $productRepository = $this->objectManager->create(ProductRepositoryInterface::class);
        $product = $productRepository->get('simple');
        $productSku = $product->getSku();

        $quote = $this->getCart('test_order_1');
        $cartId = $quote->getId();
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH .  $cartId . '/items',
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $requestData = [
            'cartItem' => [
                'sku' => $productSku,
                'qty' => 7,
                'quote_id' => $cartId,
                'custom_attributes' => [
                    [
                        'attribute_code' => 'attr_one',
                        'value' => 'value_one'
                    ],
                    [
                        'attribute_code' => 'attr_two',
                        'value' => 'value_two'
                    ],
                ],
            ],
        ];
        $this->_webApiCall($serviceInfo, $requestData);
        $quote = $this->getCart('test_order_1');
        self::assertTrue($quote->hasProductId($product->getId()));
        self::assertEquals(
            [
                'attr_one' => 'value_one',
                'attr_two' => 'value_two'
            ],
            $this->customAttributeConverter->toSerializableFormat(
                $quote->getItemByProduct($product)->getCustomAttributes()
            ),
        );
    }
}
