<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\CustomAttributeSerializable\Test\Api\CreditMemo;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\ObjectManager;
use Magento\TestFramework\TestCase\WebapiAbstract;

/**
 * Tests for credit memo and credit memo item custom attributes
 */
class CreditMemoCustomAttributesTest extends WebapiAbstract
{
    const RESOURCE_PATH = '/V1/creditmemo';

    private const CODE_ONE = 'attr_one';
    private const CODE_TWO = 'attr_two';
    private const CODE_THREE = 'attr_three';
    private const VALUE_ONE = 'value_one';
    private const VALUE_ONE_UPDATED = 'value_one_updated';
    private const VALUE_TWO = 'value_two';
    private const VALUE_TWO_UPDATED = 'value_two_updated';
    private const VALUE_THREE = 'value_three';

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customAttributeConverter = $this->objectManager->create(CustomAttributeConverter::class);
    }

    /**
     * Tests credit memo and credit memo item custom attributes are contained in a response to a GET request
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_creditmemo_with_two_items.php
     */
    public function testGetCreditMemoWithCustomAttributes()
    {
        /** @var Order $order */
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');
        $creditMemo = $order->getCreditmemosCollection()->getFirstItem();
        $creditMemo->setCustomAttribute(self::CODE_ONE, self::VALUE_ONE);
        $creditMemo->setCustomAttribute(self::CODE_TWO, self::VALUE_TWO);
        foreach ($creditMemo->getItems() as $creditMemoItem) {
            $creditMemoItem->setCustomAttribute(self::CODE_ONE, self::VALUE_ONE);
            $creditMemoItem->setCustomAttribute(self::CODE_TWO, self::VALUE_TWO);
        }

        $creditMemoRepository = $this->objectManager->get(CreditmemoRepositoryInterface::class);
        $creditMemoRepository->save($creditMemo);

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH . '/' . $creditMemo->getId(),
                'httpMethod' => Request::HTTP_METHOD_GET,
            ],
        ];

        $creditMemoResponseData = $this->_webApiCall($serviceInfo, ['id' => $creditMemo->getId()]);
        self::assertEquals($this->getCustomAttributes(), $creditMemoResponseData['custom_attributes'] ?? []);
        self::assertArrayHasKey('items', $creditMemoResponseData);
        foreach ($creditMemoResponseData['items'] as $item) {
            self::assertEquals($this->getCustomAttributes(), $item['custom_attributes'] ?? []);
        }
    }

    /**
     * Tests that credit memo and credit memo item custom attributes are saved via REST API
     *
     * @magentoApiDataFixture Magento/Sales/_files/order.php
     */
    public function testCreateAndUpdateCustomAttributesOnCreditMemo()
    {
        /** @var Order $order */
        $order = $this->objectManager->create(Order::class)->loadByIncrementId('100000001');
        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::RESOURCE_PATH,
                'httpMethod' => Request::HTTP_METHOD_POST,
            ],
        ];

        $orderItem = $order->getAllItems()[0];
        $data = [
            'order_id' => $order->getId(),
            'items' => [
                [
                    'order_item_id' => $orderItem->getId(),
                    'qty' => $orderItem->getQtyOrdered(),
                    'base_price' => $orderItem->getBasePrice(),
                    'base_cost' => $orderItem->getBaseCost(),
                    'custom_attributes' => $this->getCustomAttributes()
                ],
            ],
            'custom_attributes' => $this->getCustomAttributes(),
        ];
        $response = $this->_webApiCall($serviceInfo, ['entity' => $data]);
        self::assertArrayHasKey('entity_id', $response);
        self::assertArrayHasKey('custom_attributes', $response);
        $this->assertCreditMemoAttributes(
            $response['entity_id'],
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_TWO => self::VALUE_TWO,
            ]
        );

        $data['entity_id'] = $response['entity_id'];
        $data['custom_attributes'] = $this->getCustomAttributesUpdated();
        $data['items'][0]['custom_attributes'] = $this->getCustomAttributesUpdated();
        $data['items'][0]['entity_id'] = $response['items'][0]['entity_id'];
        $updateResponse = $this->_webApiCall($serviceInfo, ['entity' => $data]);
        self::assertArrayHasKey('entity_id', $updateResponse);
        self::assertArrayHasKey('custom_attributes', $updateResponse);
        $this->assertCreditMemoAttributes(
            $updateResponse['entity_id'],
            [
                self::CODE_ONE => self::VALUE_ONE_UPDATED,
                self::CODE_TWO => self::VALUE_TWO_UPDATED,
                self::CODE_THREE => self::VALUE_THREE,
            ]
        );
    }

    /**
     * Asserts that credit memo custom attributes are saved
     *
     * @param int $creditMemoId
     * @param array $expectedAttributes
     * @return void
     */
    private function assertCreditMemoAttributes(int $creditMemoId, array $expectedAttributes): void
    {
        $creditMemoRepository = $this->objectManager->create(CreditmemoRepositoryInterface::class);
        $creditMemo = $creditMemoRepository->get($creditMemoId);
        self::assertEquals(
            $expectedAttributes,
            $this->customAttributeConverter->toSerializableFormat($creditMemo->getCustomAttributes())
        );
        foreach ($creditMemo->getItems() as $creditMemoItem) {
            self::assertEquals(
                $expectedAttributes,
                $this->customAttributeConverter->toSerializableFormat($creditMemoItem->getCustomAttributes())
            );
        }
    }

    /**
     * Returns custom attributes
     *
     * @return array
     */
    private function getCustomAttributes(): array
    {
        return [
            [
                'attribute_code' => self::CODE_ONE,
                'value' => self::VALUE_ONE
            ],
            [
                'attribute_code' => self::CODE_TWO,
                'value' => self::VALUE_TWO
            ],
        ];
    }

    /**
     * Returns a set of custom attributes with updated values
     *
     * @return array
     */
    private function getCustomAttributesUpdated(): array
    {
        return [
            [
                'attribute_code' => self::CODE_ONE,
                'value' => self::VALUE_ONE_UPDATED
            ],
            [
                'attribute_code' => self::CODE_TWO,
                'value' => self::VALUE_TWO_UPDATED
            ],
            [
                'attribute_code' => self::CODE_THREE,
                'value' => self::VALUE_THREE
            ]
        ];
    }
}
