<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializable\Observer;

use Magento\CustomAttributeSerializable\Model\AttributesConfigurationPool;
use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Sales\Api\OrderRepositoryInterface;

/**
 * Merges existed custom attributes with new ones before order save.
 *
 * Without this observer if order is updated via Rest API the custom attributes will be overridden and not merged
 */
class OrderCustomAttributeSerializableSaveBefore implements ObserverInterface
{
    /**
     * @param OrderRepositoryInterface $orderRepository
     * @param CustomAttributeConverter $attributeConverter
     */
    public function __construct(
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly CustomAttributeConverter $attributeConverter
    ) {
    }

    /**
     * Merges existed custom attributes with new ones before order save.
     *
     * @param Observer $observer
     * @return void
     */
    public function execute(Observer $observer)
    {
        $orderToUpdate = $observer->getEvent()->getDataObject();

        if ($orderToUpdate->isObjectNew()
            || !$orderToUpdate->hasData(AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE)
            || empty($orderToUpdate->getId())
        ) {
            return;
        }

        $currentOrder = $this->orderRepository->get($orderToUpdate->getId());

        $customAttributes = $this->mergeCustomAttributes(
            $currentOrder->getCustomAttributes() ?? [],
            $orderToUpdate->getCustomAttributes() ?? []
        );

        $orderToUpdate->setCustomAttributes($customAttributes);
        $orderToUpdate->setData(
            AttributesConfigurationPool::CUSTOM_ATTRIBUTES_SERIALIZABLE,
            $this->attributeConverter->toSerializableFormat($customAttributes)
        );
    }

    /**
     * Merges the custom attributes from multiple arrays into a single array
     *
     * @param array $customAttributes
     * @return array
     */
    private function mergeCustomAttributes(array ...$customAttributes): array
    {
        $attributes = [];
        foreach ($customAttributes as $customAttribute) {
            foreach ($customAttribute as $attribute) {
                $attributes[$attribute->getAttributeCode()] = $attribute;
            }
        }

        return $attributes;
    }
}
