<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\Quote;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Quote\Api\CartRepositoryInterfaceFactory;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\Quote\Model\QuoteIdMaskFactory;
use Magento\Quote\Model\ResourceModel\Quote as QuoteResource;
use Magento\Quote\Model\ResourceModel\Quote\CollectionFactory as QuoteCollectionFactory;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Test the assigning and updating custom attributes on cart
 */
class SetCustomAttributesOnCartTest extends GraphQlAbstract
{
    /**
     * @var CartRepositoryInterfaceFactory
     */
    private $cartRepositoryFactory;

    /**
     * @var GetCustomerAuthenticationHeader
     */
    private $getCustomerAuthenticationHeader;

    /**
     * @var QuoteCollectionFactory
     */
    private $quoteCollectionFactory;

    /**
     * @var QuoteResource
     */
    private $quoteResource;

    /**
     * @var QuoteIdMaskFactory
     */
    private $quoteIdMaskFactory;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface|mixed
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->maskedQuoteIdToQuoteId = $objectManager->get(MaskedQuoteIdToQuoteIdInterface::class);
        $this->getCustomerAuthenticationHeader = $objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->customAttributeConverter = $objectManager->get(CustomAttributeConverter::class);
        $this->quoteCollectionFactory = $objectManager->get(QuoteCollectionFactory::class);
        $this->cartRepositoryFactory = $objectManager->get(CartRepositoryInterfaceFactory::class);
        $this->quoteResource = $objectManager->get(QuoteResource::class);
        $this->quoteIdMaskFactory = $objectManager->get(QuoteIdMaskFactory::class);
        $this->queryHelper = $objectManager->get(QueryHelper::class);
    }

    /**
     * Test for setting and updating custom attributes on cart
     *
     * @magentoApiDataFixture Magento/Customer/_files/customer.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/customer/create_empty_cart.php
     */
    public function testSetAndUpdateCartCustomAttributes()
    {
        $this->_markTestAsRestOnly();
        $customAttributes = [
            'cart attribute one' => 'value one',
            'cart attribute two' => 'value two',
        ];
        $customAttributesUpdated = [
            'cart attribute one' => 'value one updated',
            'cart attribute two' => '',
            'cart attribute three' => 'value three',
        ];

        $maskedCartId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_quote');
        $quoteId = $this->maskedQuoteIdToQuoteId->execute($maskedCartId);

        $cart = $this->setAndValidateCustomAttributesOnCart($maskedCartId, $quoteId, $customAttributes);

        self::assertNotNull($cart->getId());
        self::assertEquals(1, $cart->getCustomer()->getId());
        self::assertEquals('default', $cart->getStore()->getCode());
        self::assertEquals($customAttributes, $cart->getCustomAttributesSerializable());
        self::assertCount(2, $cart->getCustomAttributes());
        self::assertEquals(
            $customAttributes,
            $this->customAttributeConverter->toSerializableFormat($cart->getCustomAttributes())
        );

        $cart = $this->setAndValidateCustomAttributesOnCart($maskedCartId, $quoteId, $customAttributesUpdated);

        self::assertEquals(array_filter($customAttributesUpdated), $cart->getCustomAttributesSerializable());
        self::assertCount(2, $cart->getCustomAttributes());
        self::assertEquals(
            array_filter($customAttributesUpdated),
            $this->customAttributeConverter->toSerializableFormat($cart->getCustomAttributes())
        );
    }

    /**
     * Updates and validates custom attributes on cart, and returns the cart entity
     *
     * @param string $maskedCartId
     * @param int $quoteId
     * @param array $customAttributesUpdated
     * @return CartInterface
     * @throws AuthenticationException
     * @throws NoSuchEntityException
     */
    private function setAndValidateCustomAttributesOnCart(
        string $maskedCartId,
        int $quoteId,
        array $customAttributesUpdated
    ): CartInterface {
        $updateCustomAttributesResponse = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCartMutationQuery($maskedCartId, $customAttributesUpdated),
            [],
            '',
            $this->getCustomerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('setCustomAttributesOnCart', $updateCustomAttributesResponse);
        self::assertArrayHasKey('cart', $updateCustomAttributesResponse['setCustomAttributesOnCart']);
        self::assertArrayHasKey(
            'custom_attributes',
            $updateCustomAttributesResponse['setCustomAttributesOnCart']['cart']
        );

        $updateCustomAttributesResponse = $updateCustomAttributesResponse['setCustomAttributesOnCart']['cart'];
        self::assertCount(2, $updateCustomAttributesResponse['custom_attributes']);

        $cartRepository = $this->cartRepositoryFactory->create();
        return $cartRepository->get($quoteId);
    }

    /**
     * @return void
     * @throws \Exception
     */
    protected function tearDown(): void
    {
        $quoteCollection = $this->quoteCollectionFactory->create();
        foreach ($quoteCollection as $quote) {
            $this->quoteResource->delete($quote);

            $quoteIdMask = $this->quoteIdMaskFactory->create();
            $quoteIdMask->setQuoteId($quote->getId())
                ->delete();
        }
        parent::tearDown();
    }
}
