<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Model\Resolver\Cart\Item;

use Magento\CustomAttributeSerializableGraphQl\Model\Resolver\InputArgumentValidator;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\QuoteGraphQl\Model\Cart\GetCartForUser;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;

/**
 * Sets custom attributes on a cart item
 */
class SetCustomAttributesOnCartItem implements ResolverInterface
{
    /**
     * @param GetCartForUser $getCartForUser
     * @param CartRepositoryInterface $cartRepository
     * @param Uid $uidEncoder
     * @param InputArgumentValidator $inputArgumentValidator
     */
    public function __construct(
        private readonly GetCartForUser $getCartForUser,
        private readonly CartRepositoryInterface $cartRepository,
        private readonly Uid $uidEncoder,
        private readonly InputArgumentValidator $inputArgumentValidator
    ) {
    }

    /**
     * Sets custom attributes on a cart item
     *
     * @param Field $field
     * @param mixed $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     * @throws GraphQlAuthorizationException
     * @throws GraphQlNoSuchEntityException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, ?array $value = null, ?array $args = null)
    {
        $inputArgs = $args['input'];
        $this->inputArgumentValidator->validate($inputArgs, ['cart_id', 'cart_item_id'], ['cart_item_id']);

        try {
            $cartItemId = $this->uidEncoder->decode($inputArgs['cart_item_id']);

            $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();

            // Shopping Cart validation
            $cart = $this->getCartForUser->execute($inputArgs['cart_id'], $context->getUserId(), $storeId);
        } catch (GraphQlNoSuchEntityException $e) {
            throw $e;
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__('The cart can\'t be loaded.'));
        }

        foreach ($cart->getItems() as $item) {
            if ($item->getItemId() == $cartItemId) {
                $attrs = [];
                foreach ($inputArgs['custom_attributes'] as $attribute) {
                    $attrs[$attribute['attribute_code']] = $attribute['value'];
                }
                $item->setCustomAttributes($attrs);
                $this->cartRepository->save($cart);

                return [
                    'cart' => [
                        'model' => $cart
                    ]
                ];
            }
        }

        throw new GraphQlInputException(__('The cart doesn\'t have an item with the provided cartItemId'));
    }
}
