<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\CreditMemo;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\FunctionalTestingFramework\ObjectManager;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Api\Data\CreditmemoInterface;
use Magento\Sales\Model\Order;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Tests the assigning and updating of custom attributes on credit memos
 */
class SetCustomAttributesOnCreditMemoTest extends GraphQlAbstract
{
    private const CODE_ONE = 'credit_memo_code_one';
    private const CODE_TWO = 'credit_memo_code_two';
    private const CODE_THREE = 'credit_memo_code_three';
    private const VALUE_ONE = 'credit_memo_value_one';
    private const VALUE_TWO = 'credit_memo_value_two';
    private const VALUE_THREE = 'credit_memo_value_three';

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var GetCustomerAuthenticationHeader
     */
    private $customerAuthenticationHeader;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    /**
     * @var Uid
     */
    private $uid;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customerAuthenticationHeader = $this->objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->customAttributeConverter = $this->objectManager->get(CustomAttributeConverter::class);
        $this->queryHelper = $this->objectManager->get(QueryHelper::class);
        $this->uid = $this->objectManager->get(Uid::class);
    }

    /**
     * Tests that custom attributes can be retrieved for a credit memo
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_creditmemo_with_two_items.php
     */
    public function testGetCustomAttributesForCreditMemo()
    {
        $this->_markTestAsRestOnly();
        $customAttributes = [
            self::CODE_ONE => self::VALUE_ONE,
            self::CODE_TWO => self::VALUE_TWO,
        ];

        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var CreditmemoInterface $creditMemo */
        $creditMemo = $order->getCreditmemosCollection()->getFirstItem();
        $creditMemo->setCustomAttributes($customAttributes);
        $this->objectManager->create(CreditmemoRepositoryInterface::class)->save($creditMemo);

        $response = $this->graphQlQuery(
            $this->queryHelper->getOrderWithCreditMemoAndCreditMemoItems('100000001'),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('customer', $response);
        self::assertArrayHasKey('orders', $response['customer']);
        self::assertArrayHasKey('items', $response['customer']['orders']);
        $creditMemos = array_shift($response['customer']['orders']['items']);
        $creditMemo = array_shift($creditMemos['credit_memos']);
        self::assertArrayHasKey('custom_attributes', $creditMemo);

        foreach ($creditMemo['custom_attributes'] as $attribute) {
            self::assertArrayHasKey('attribute_code', $attribute);
            self::assertArrayHasKey('value', $attribute);
            self::assertArrayHasKey($attribute['attribute_code'], $customAttributes);
        }
    }

    /**
     * Tests that custom attributes can be set and updated for a credit memo
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_creditmemo_with_two_items.php
     */
    public function testSetCustomAttributesOnCreditMemo()
    {
        $this->_markTestAsRestOnly();
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var CreditmemoInterface $creditMemo */
        $creditMemo = $order->getCreditmemosCollection()->getFirstItem();

        $initialAttributes = [
            self::CODE_ONE => self::VALUE_ONE,
            self::CODE_TWO => self::VALUE_TWO,
        ];
        $this->setCustomAttributesAndValidate(
            $creditMemo,
            $initialAttributes,
            $initialAttributes
        );

        $this->setCustomAttributesAndValidate(
            $creditMemo,
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_TWO => '',
                self::CODE_THREE => self::VALUE_THREE,
            ],
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_THREE => self::VALUE_THREE,
            ]
        );
    }

    /**
     *  Sets custom attributes on a credit memo using a GraphQl mutation and validates the saving of the attributes
     *
     * @param CreditmemoInterface $creditMemo
     * @param array $customAttributes
     * @param array $expectedAttributes
     * @return void
     * @throws \Magento\Framework\Exception\AuthenticationException
     */
    private function setCustomAttributesAndValidate(
        CreditmemoInterface $creditMemo,
        array $customAttributes,
        array $expectedAttributes
    ) {
        $response = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCreditMemoMutationQuery(
                $this->uid->encode($creditMemo->getId()),
                $customAttributes
            ),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('setCustomAttributesOnCreditMemo', $response);
        self::assertArrayHasKey('credit_memo', $response['setCustomAttributesOnCreditMemo']);
        $responseCreditMemo = $response['setCustomAttributesOnCreditMemo']['credit_memo'];
        self::assertCount(count($expectedAttributes), $responseCreditMemo['custom_attributes']);

        $creditMemo = $this->objectManager->create(CreditmemoRepositoryInterface::class)->get($creditMemo->getId());
        self::assertEquals(
            $expectedAttributes,
            $this->customAttributeConverter->toSerializableFormat($creditMemo->getCustomAttributes())
        );
    }
}
