<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Model\Resolver\Invoice;

use Magento\CustomAttributeSerializableGraphQl\Model\Resolver\InputArgumentValidator;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Sales\Api\InvoiceRepositoryInterface;

/**
 * Sets custom attributes on invoice
 */
class SetCustomAttributesOnInvoice implements ResolverInterface
{
    /**
     * @param InvoiceRepositoryInterface $invoiceRepository
     * @param InvoiceResponse $invoiceResponse
     * @param Uid $uidEncoder
     * @param InputArgumentValidator $inputArgumentValidator
     */
    public function __construct(
        private readonly InvoiceRepositoryInterface $invoiceRepository,
        private readonly InvoiceResponse $invoiceResponse,
        private readonly Uid $uidEncoder,
        private readonly InputArgumentValidator $inputArgumentValidator,
    ) {
    }

    /**
     * Sets custom attributes on invoice
     *
     * @param Field $field
     * @param mixed $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     * @throws GraphQlAuthorizationException
     * @throws GraphQlNoSuchEntityException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, ?array $value = null, ?array $args = null)
    {
        $inputArgs = $args['input'];
        $this->inputArgumentValidator->validate($inputArgs, ['invoice_id'], ['invoice_id']);

        try {
            $invoiceId = $this->uidEncoder->decode($inputArgs['invoice_id']);
            $invoice = $this->invoiceRepository->get($invoiceId);

            if ((int)$invoice->getOrder()->getCustomerId() !== (int)$context->getUserId()) {
                throw new GraphQlAuthorizationException(__('The invoice doesn\'t belong to the customer.'));
            }

            foreach ($inputArgs['custom_attributes'] as $attribute) {
                $invoice->setCustomAttribute($attribute['attribute_code'], $attribute['value']);
            }
            $this->invoiceRepository->save($invoice);

            return $this->invoiceResponse->create($invoice);
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__('The invoice doesn\'t exist.'));
        } catch (CouldNotSaveException $e) {
            throw new GraphQlInputException(__('Could not save the invoice.'));
        }
    }
}
