<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Model\Resolver\CreditMemo\Item;

use Magento\CustomAttributeSerializableGraphQl\Model\Resolver\InputArgumentValidator;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Sales\Api\CreditmemoItemRepositoryInterface;
use Magento\Sales\Api\CreditmemoRepositoryInterface;

/**
 * Sets custom attributes on a credit memo item
 */
class SetCustomAttributesOnCreditMemoItem implements ResolverInterface
{
    /**
     * @param CreditmemoRepositoryInterface $creditMemoRepository
     * @param CreditmemoItemRepositoryInterface $creditMemoItemRepository
     * @param Uid $uidEncoder
     * @param InputArgumentValidator $argumentValidator
     */
    public function __construct(
        private readonly CreditmemoRepositoryInterface $creditMemoRepository,
        private readonly CreditmemoItemRepositoryInterface $creditMemoItemRepository,
        private readonly Uid $uidEncoder,
        private readonly InputArgumentValidator $argumentValidator
    ) {
    }

    /**
     * Sets custom attributes on a credit memo item
     *
     * @param Field $field
     * @param mixed $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array
     * @throws GraphQlInputException
     * @throws GraphQlAuthorizationException
     * @throws GraphQlNoSuchEntityException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(Field $field, $context, ResolveInfo $info, ?array $value = null, ?array $args = null)
    {
        $inputArgs = $args['input'];
        $idFields = ['credit_memo_id', 'credit_memo_item_id'];
        $this->argumentValidator->validate($inputArgs, $idFields, $idFields);

        try {
            $creditMemoId = $this->uidEncoder->decode($inputArgs['credit_memo_id']);
            $creditMemoItemId = $this->uidEncoder->decode($inputArgs['credit_memo_item_id']);
            $creditMemoItem = $this->creditMemoItemRepository->get($creditMemoItemId);
            if ((int)$creditMemoItem->getEntityId() === 0) {
                throw new GraphQlNoSuchEntityException(__('The credit memo item doesn\'t exist.'));
            }

            $creditMemo = $this->creditMemoRepository->get($creditMemoItem->getParentId());

            if ((int)$creditMemo->getId() !== (int)$creditMemoId) {
                throw new GraphQlNoSuchEntityException(__('The credit memo item doesn\'t belong to the credit memo.'));
            }

            if ((int)$creditMemo->getOrder()->getCustomerId() !== (int)$context->getUserId()) {
                throw new GraphQlAuthorizationException(__('The credit memo doesn\'t belong to the customer.'));
            }

            $attrs = [];
            foreach ($inputArgs['custom_attributes'] as $attribute) {
                $attrs[$attribute['attribute_code']] = $attribute['value'];
            }
            $creditMemoItem->setCustomAttributes($attrs);
            $this->creditMemoItemRepository->save($creditMemoItem);

            return [
                'credit_memo' => [
                    'id' => base64_encode((string)$creditMemo->getId()),
                    'number' => $creditMemo->getIncrementId(),
                    'model' => $creditMemo,
                    'order' => $creditMemo->getOrder()
                ]
            ];
        } catch (NoSuchEntityException $e) {
            throw new GraphQlNoSuchEntityException(__('The credit memo item doesn\'t exist.'));
        } catch (CouldNotSaveException $e) {
            throw new GraphQlInputException(__('Could not save the credit memo item.'));
        }
    }
}
