<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\Quote;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Quote\Api\CartRepositoryInterfaceFactory;
use Magento\Quote\Api\Data\CartItemInterface;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Test the assigning and updating custom attributes on cart item
 */
class SetCustomAttributesOnCartItemTest extends GraphQlAbstract
{
    /**
     * @var CartRepositoryInterfaceFactory
     */
    private $cartRepositoryFactory;

    /**
     * @var GetCustomerAuthenticationHeader
     */
    private $getCustomerAuthenticationHeader;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface|mixed
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->maskedQuoteIdToQuoteId = $objectManager->get(MaskedQuoteIdToQuoteIdInterface::class);
        $this->getCustomerAuthenticationHeader = $objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->customAttributeConverter = $objectManager->get(CustomAttributeConverter::class);
        $this->cartRepositoryFactory = $objectManager->get(CartRepositoryInterfaceFactory::class);
        $this->queryHelper = $objectManager->get(QueryHelper::class);
    }

    /**
     * @magentoApiDataFixture Magento/Customer/_files/customer.php
     * @magentoApiDataFixture Magento/GraphQl/Catalog/_files/simple_product.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/customer/create_empty_cart.php
     */
    public function testCreateEmptyCartItemSetAndUpdateCustomAttributes()
    {
        $customAttributes = [
            'cart item attribute one' => 'value one',
            'cart item attribute two' => 'value two',
        ];
        $customAttributesUpdated = [
            'cart item attribute one' => 'value one updated',
            'cart item attribute two' => '',
            'cart item attribute three' => 'value three',
        ];

        $sku = 'simple_product';
        $maskedCartId = $this->getMaskedQuoteIdByReservedOrderId->execute('test_quote');

        $query = $this->getAddProductQuery($maskedCartId, $sku, 2);
        $addProductResponse = $this->graphQlMutation($query, [], '', $this->getCustomerAuthenticationHeader->execute());

        self::assertArrayHasKey('addProductsToCart', $addProductResponse);
        self::assertEquals(
            $sku,
            $addProductResponse['addProductsToCart']['cart']['itemsV2']['items'][0]['product']['sku']
        );

        $quoteItemUid = $addProductResponse['addProductsToCart']['cart']['itemsV2']['items'][0]['uid'];

        $cartItem = $this->setAndValidateCustomAttributesOnCartItem($maskedCartId, $quoteItemUid, $customAttributes);
        self::assertEquals(
            $customAttributes,
            $this->customAttributeConverter->toSerializableFormat($cartItem->getCustomAttributes())
        );

        $cartItem = $this->setAndValidateCustomAttributesOnCartItem(
            $maskedCartId,
            $quoteItemUid,
            $customAttributesUpdated,
        );
        self::assertEquals(
            array_filter($customAttributesUpdated),
            $this->customAttributeConverter->toSerializableFormat($cartItem->getCustomAttributes())
        );
    }

    /**
     * Updates and validates custom attributes on cart item, and returns the cart item entity
     *
     * @param string $maskedCartId
     * @param mixed $quoteItemUid
     * @param array $customAttributesUpdated
     * @return CartItemInterface
     * @throws AuthenticationException
     * @throws NoSuchEntityException
     */
    public function setAndValidateCustomAttributesOnCartItem(
        string $maskedCartId,
        mixed  $quoteItemUid,
        array  $customAttributesUpdated,
    ): CartItemInterface {
        $updateAttributesResponse = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCartItemMutationQuery(
                $maskedCartId,
                $quoteItemUid,
                $customAttributesUpdated
            ),
            [],
            '',
            $this->getCustomerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('setCustomAttributesOnCartItem', $updateAttributesResponse);

        $cartData = $updateAttributesResponse['setCustomAttributesOnCartItem']['cart'];
        $attributes = $cartData['itemsV2']['items'][0]['custom_attributes'];
        self::assertCount(2, $attributes);

        $cartRepository = $this->cartRepositoryFactory->create();
        $quoteId = $this->maskedQuoteIdToQuoteId->execute($maskedCartId);
        $cart = $cartRepository->get($quoteId);

        $cartItem = $cart->getItems()[0];
        self::assertCount(2, $cartItem->getCustomAttributes());

        return $cartItem;
    }

    /**
     * Returns the query to add a simple product to the cart
     *
     * @param string $maskedQuoteId
     * @param string $sku
     * @param int $quantity
     * @return string
     */
    private function getAddProductQuery(string $maskedQuoteId, string $sku, int $quantity): string
    {
        return <<<QUERY
mutation {
  addProductsToCart(
    cartId: "{$maskedQuoteId}",
    cartItems: [
      {
        quantity: {$quantity}
        sku: "{$sku}"
      }
    ],
  ) {
    cart {
      itemsV2 {
        items {
          uid
          quantity
          product {
            sku
          }
        }
      }
    }
  }
}
QUERY;
    }
}
