<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\Order;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Api\SearchCriteria;
use Magento\Framework\Registry;
use Magento\CustomAttributeSerializableGraphQl\Test\Api\Quote\QueryHelper;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\GraphQl\Quote\GetMaskedQuoteIdByReservedOrderId;
use Magento\Quote\Api\CartRepositoryInterfaceFactory;
use Magento\Quote\Model\MaskedQuoteIdToQuoteIdInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Test of the copying custom attributes from quote to order and from quote item to order item
 */
class CreateOrderFromQuoteWithCustomAttributes extends GraphQlAbstract
{
    /**
     * @var GetMaskedQuoteIdByReservedOrderId
     */
    private $getMaskedQuoteIdByReservedOrderId;

    /**
     * @var CollectionFactory
     */
    private $orderCollectionFactory;

    /**
     * @var OrderRepositoryInterface
     */
    private $orderRepository;

    /**
     * @var Registry
     */
    private $registry;

    /**
     * @var GetCustomerAuthenticationHeader
     */
    private $getCustomerAuthenticationHeader;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    /**
     * @var CartRepositoryInterfaceFactory
     */
    private $cartRepositoryFactory;

    /**
     * @var MaskedQuoteIdToQuoteIdInterface
     */
    private $maskedQuoteIdToQuoteId;

    /**
     * @inheritdoc
     */
    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->getMaskedQuoteIdByReservedOrderId = $objectManager->get(GetMaskedQuoteIdByReservedOrderId::class);
        $this->orderCollectionFactory = $objectManager->get(CollectionFactory::class);
        $this->orderRepository = $objectManager->get(OrderRepositoryInterface::class);
        $this->registry = Bootstrap::getObjectManager()->get(Registry::class);
        $this->getCustomerAuthenticationHeader = $objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->customAttributeConverter = $objectManager->get(CustomAttributeConverter::class);
        $this->cartRepositoryFactory = $objectManager->get(CartRepositoryInterfaceFactory::class);
        $this->queryHelper = $objectManager->get(QueryHelper::class);
        $this->maskedQuoteIdToQuoteId = $objectManager->get(MaskedQuoteIdToQuoteIdInterface::class);
    }

    /**
     * @magentoApiDataFixture Magento/Customer/_files/customer.php
     * @magentoApiDataFixture Magento/GraphQl/Catalog/_files/simple_product.php
     * @magentoConfigFixture default_store carriers/flatrate/active 1
     * @magentoConfigFixture default_store carriers/tablerate/active 1
     * @magentoConfigFixture default_store carriers/freeshipping/active 1
     * @magentoConfigFixture default_store payment/banktransfer/active 1
     * @magentoConfigFixture default_store payment/cashondelivery/active 1
     * @magentoConfigFixture default_store payment/checkmo/active 1
     * @magentoConfigFixture default_store payment/purchaseorder/active 1
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/customer/create_empty_cart.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/add_simple_product.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/set_new_shipping_address.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/set_new_billing_address.php
     * @magentoApiDataFixture Magento/GraphQl/Quote/_files/set_flatrate_shipping_method.php
     */
    public function testSetPaymentMethodAndPlaceOrderFromQuoteWithCustomAttributes()
    {
        $cartCustomAttributes = [
            'cart attribute one' => 'value one',
            'cart attribute two' => 'value two',
        ];
        $cartItemCustomAttributes = [
            'cart item attribute one' => 'value one',
            'cart item attribute two' => 'value two',
        ];
        $reservedOrderId = 'test_quote';
        $maskedCartId = $this->getMaskedQuoteIdByReservedOrderId->execute($reservedOrderId);
        $quoteId = $this->maskedQuoteIdToQuoteId->execute($maskedCartId);

        $updateCustomAttributesResponse = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCartMutationQuery($maskedCartId, $cartCustomAttributes),
            [],
            '',
            $this->getCustomerAuthenticationHeader->execute()
        );
        $cartRepository = $this->cartRepositoryFactory->create();
        $cart = $cartRepository->get($quoteId);

        self::assertArrayHasKey('setCustomAttributesOnCart', $updateCustomAttributesResponse);
        self::assertArrayHasKey('cart', $updateCustomAttributesResponse['setCustomAttributesOnCart']);
        self::assertArrayHasKey(
            'custom_attributes',
            $updateCustomAttributesResponse['setCustomAttributesOnCart']['cart']
        );
        $updateCustomAttributesResponse = $updateCustomAttributesResponse['setCustomAttributesOnCart']['cart'];
        self::assertCount(2, $updateCustomAttributesResponse['custom_attributes']);
        self::assertEquals(
            $cartCustomAttributes,
            $this->customAttributeConverter->toSerializableFormat($cart->getCustomAttributes())
        );
        $cartItem = $cart->getItems()[0];
        $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCartItemMutationQuery(
                $maskedCartId,
                base64_encode($cartItem->getId()),
                $cartItemCustomAttributes
            ),
            [],
            '',
            $this->getCustomerAuthenticationHeader->execute()
        );

        $paymentMethod = 'checkmo';
        $setPaymentMethodQuery = $this->getSetPaymentMethodQuery($maskedCartId, $paymentMethod);
        $setPaymentMethodQueryResponse = $this->graphQlMutation(
            $setPaymentMethodQuery,
            [],
            '',
            $this->getCustomerAuthenticationHeader->execute()
        );
        self::assertArrayHasKey('setPaymentMethodOnCart', $setPaymentMethodQueryResponse);
        self::assertArrayHasKey(
            'available_payment_methods',
            $setPaymentMethodQueryResponse['setPaymentMethodOnCart']['cart']
        );
        self::assertEquals(
            $paymentMethod,
            $setPaymentMethodQueryResponse['setPaymentMethodOnCart']['cart']['selected_payment_method']['code'] ?? ''
        );

        $query = $this->getPlaceOrderQuery($maskedCartId);
        $response = $this->graphQlMutation($query, [], '', $this->getCustomerAuthenticationHeader->execute());

        self::assertArrayHasKey('placeOrder', $response);
        self::assertArrayHasKey('order_number', $response['placeOrder']['order']);
        self::assertEquals($reservedOrderId, $response['placeOrder']['order']['order_number']);

        $order = $this->orderRepository->getList(new SearchCriteria(['quote_id' => $quoteId]))->getFirstItem();
        self::assertEquals(
            $cartCustomAttributes,
            $this->customAttributeConverter->toSerializableFormat($order->getCustomAttributes())
        );
        $orderItem = array_first($order->getItems());
        self::assertEquals(
            $cartItemCustomAttributes,
            $this->customAttributeConverter->toSerializableFormat($orderItem->getCustomAttributes())
        );
    }

    /**
     * Returns query for setting payment method
     *
     * @param string $maskedQuoteId
     * @param string $paymentMethod
     * @return string
     */
    private function getSetPaymentMethodQuery(string $maskedQuoteId, string $paymentMethod): string
    {
        return <<<QUERY
mutation {
  setPaymentMethodOnCart(
    input: {
      cart_id: "{$maskedQuoteId}"
      payment_method: { code: "{$paymentMethod}" }
    }
  ) {
    cart {
      available_payment_methods {
        code
        title
      }
      selected_payment_method {
        code
      }
    }
  }
}
QUERY;
    }

    /**
     * Returns query for placing order
     *
     * @param string $maskedQuoteId
     * @return string
     */
    private function getPlaceOrderQuery(string $maskedQuoteId): string
    {
        return <<<QUERY
mutation {
  placeOrder(input: {cart_id: "{$maskedQuoteId}"}) {
    order {
      order_number
    }
  }
}
QUERY;
    }

    /**
     * @inheritdoc
     */
    protected function tearDown(): void
    {
        $this->registry->unregister('isSecureArea');
        $this->registry->register('isSecureArea', true);

        $orderCollection = $this->orderCollectionFactory->create();
        foreach ($orderCollection as $order) {
            $this->orderRepository->delete($order);
        }
        $this->registry->unregister('isSecureArea');
        $this->registry->register('isSecureArea', false);

        parent::tearDown();
    }
}
