<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\Invoice;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\ObjectManagerInterface;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\Sales\Api\Data\InvoiceInterface;
use Magento\Sales\Api\InvoiceRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Test the assigning and updating custom attributes on invoice
 */
class SetCustomAttributesOnInvoiceTest extends GraphQlAbstract
{
    /** @var GetCustomerAuthenticationHeader */
    private $customerAuthenticationHeader;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    /**
     * @var Uid
     */
    private $uid;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customerAuthenticationHeader = $this->objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->queryHelper = $this->objectManager->get(QueryHelper::class);
        $this->uid = $this->objectManager->get(Uid::class);
        $this->customAttributeConverter = $this->objectManager->get(CustomAttributeConverter::class);
    }

    /**
     * Tests that custom attributes can be received for invoice
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_invoice_with_two_products_and_custom_options.php
     */
    public function testGetCustomAttributesForInvoice()
    {
        $customAttributes = [
            'invoice attribute one' => 'value one',
            'invoice attribute two' => 'value two',
        ];
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var InvoiceInterface $invoice */
        $invoice = $order->getInvoiceCollection()->getFirstItem();
        $invoice->setCustomAttributes($customAttributes);
        $this->objectManager->create(InvoiceRepositoryInterface::class)->save($invoice);

        $response = $this->graphQlQuery(
            $this->queryHelper->getOrderWithInvoiceAndInvoiceItems('100000001'),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('customer', $response);
        self::assertArrayHasKey('orders', $response['customer']);
        self::assertArrayHasKey('items', $response['customer']['orders']);
        $invoices = array_shift($response['customer']['orders']['items']);
        $invoice = array_shift($invoices['invoices']);
        self::assertArrayHasKey('custom_attributes', $invoice);

        foreach ($invoice['custom_attributes'] as $attribute) {
            self::assertArrayHasKey('attribute_code', $attribute);
            self::assertArrayHasKey('value', $attribute);
            self::assertArrayHasKey($attribute['attribute_code'], $customAttributes);
        }
    }

    /**
     * Tests that custom attributes can be set and updated on invoice
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_invoice_with_two_products_and_custom_options.php
     */
    public function testSetCustomAttributesOnInvoice()
    {
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var InvoiceInterface $invoice */
        $invoice = $order->getInvoiceCollection()->getFirstItem();

        $this->setCustomAttributesAndValidate($invoice, [
            'invoice attribute one' => 'value one',
            'invoice attribute two' => 'value two',
        ]);

        $this->setCustomAttributesAndValidate(
            $invoice,
            [
                'invoice attribute one' => 'value one',
                'invoice attribute two' => '',
                'invoice attribute three' => 'value three',
            ],
            [
                'invoice attribute one' => 'value one',
                'invoice attribute three' => 'value three',
            ]
        );
    }

    /**
     * Sets custom attributes on invoice and validates the result
     *
     * @param InvoiceInterface $invoice
     * @param array $customAttributes
     * @param array $expectedAttributes
     * @return void
     * @throws AuthenticationException
     */
    public function setCustomAttributesAndValidate(
        InvoiceInterface $invoice,
        array $customAttributes,
        array $expectedAttributes = []
    ): void {
        $updateCustomAttributesResponse = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnInvoiceMutationQuery(
                $this->uid->encode($invoice->getId()),
                $customAttributes
            ),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('setCustomAttributesOnInvoice', $updateCustomAttributesResponse);
        self::assertArrayHasKey(
            'invoice',
            $updateCustomAttributesResponse['setCustomAttributesOnInvoice']
        );
        $responseInvoice = $updateCustomAttributesResponse['setCustomAttributesOnInvoice']['invoice'];
        self::assertCount(2, $responseInvoice['custom_attributes']);

        $invoice = $this->objectManager->create(InvoiceRepositoryInterface::class)->get($invoice->getId());
        self::assertEquals(
            $expectedAttributes ?: $customAttributes,
            $this->customAttributeConverter->toSerializableFormat($invoice->getCustomAttributes())
        );
    }
}
