<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Test\Api\CreditMemo;

use Magento\CustomAttributeSerializable\Model\CustomAttributes\CustomAttributeConverter;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\FunctionalTestingFramework\ObjectManager;
use Magento\GraphQl\GetCustomerAuthenticationHeader;
use Magento\Sales\Api\CreditmemoItemRepositoryInterface;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Api\Data\CreditmemoInterface;
use Magento\Sales\Api\Data\CreditmemoItemInterface;
use Magento\Sales\Model\Order;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

/**
 * Tests the assigning and updating of custom attributes on credit memo items
 */
class SetCustomAttributesOnCreditMemoItemTest extends GraphQlAbstract
{
    private const CODE_ONE = 'credit_memo_code_one';
    private const CODE_TWO = 'credit_memo_code_two';
    private const CODE_THREE = 'credit_memo_code_three';
    private const VALUE_ONE = 'credit_memo_value_one';
    private const VALUE_TWO = 'credit_memo_value_two';
    private const VALUE_THREE = 'credit_memo_value_three';

    /**
     * @var ObjectManager
     */
    private $objectManager;

    /**
     * @var GetCustomerAuthenticationHeader
     */
    private $customerAuthenticationHeader;

    /**
     * @var CustomAttributeConverter
     */
    private $customAttributeConverter;

    /**
     * @var QueryHelper
     */
    private $queryHelper;

    /**
     * @var Uid
     */
    private $uid;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->customerAuthenticationHeader = $this->objectManager->get(GetCustomerAuthenticationHeader::class);
        $this->customAttributeConverter = $this->objectManager->get(CustomAttributeConverter::class);
        $this->queryHelper = $this->objectManager->get(QueryHelper::class);
        $this->uid = $this->objectManager->get(Uid::class);
    }

    /**
     * Tests that custom attributes can be retrieved for a credit memo item
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_creditmemo_with_two_items.php
     */
    public function testGetCustomAttributesForCreditMemoItem()
    {
        $customAttributes = [
            self::CODE_ONE => self::VALUE_ONE,
            self::CODE_TWO => self::VALUE_TWO,
        ];

        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var CreditmemoInterface $creditMemo */
        $creditMemo = $order->getCreditmemosCollection()->getFirstItem();
        foreach ($creditMemo->getItems() as $creditMemoItem) {
            $creditMemoItem->setCustomAttributes($customAttributes);
        }
        $this->objectManager->create(CreditmemoRepositoryInterface::class)->save($creditMemo);

        $response = $this->graphQlQuery(
            $this->queryHelper->getOrderWithCreditMemoAndCreditMemoItems('100000001'),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('customer', $response);
        self::assertArrayHasKey('orders', $response['customer']);
        self::assertArrayHasKey('items', $response['customer']['orders']);
        $creditMemos = array_shift($response['customer']['orders']['items']);
        $creditMemo = array_shift($creditMemos['credit_memos']);

        self::assertArrayHasKey('items', $creditMemo);
        foreach ($creditMemo['items'] as $creditMemoItem) {
            self::assertArrayHasKey('custom_attributes', $creditMemoItem);
            foreach ($creditMemoItem['custom_attributes'] as $attribute) {
                self::assertArrayHasKey('attribute_code', $attribute);
                self::assertArrayHasKey('value', $attribute);
                self::assertArrayHasKey($attribute['attribute_code'], $customAttributes);
            }
        }
    }

    /**
     * Tests that custom attributes can be set and updated for a credit memo item
     *
     * @magentoApiDataFixture Magento/Sales/_files/customer_creditmemo_with_two_items.php
     */
    public function testSetCustomAttributesOnCreditMemoItem()
    {
        $order = $this->objectManager->get(Order::class)->loadByIncrementId('100000001');

        /** @var CreditmemoInterface $creditMemo */
        $creditMemo = $order->getCreditmemosCollection()->getFirstItem();
        $creditMemoItem = array_values($creditMemo->getItems())[0];

        $initialAttributes = [
            self::CODE_ONE => self::VALUE_ONE,
            self::CODE_TWO => self::VALUE_TWO,
        ];
        $this->setCustomAttributesAndValidate(
            $creditMemo,
            $creditMemoItem,
            $initialAttributes,
            $initialAttributes
        );

        $this->setCustomAttributesAndValidate(
            $creditMemo,
            $creditMemoItem,
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_TWO => '',
                self::CODE_THREE => self::VALUE_THREE,
            ],
            [
                self::CODE_ONE => self::VALUE_ONE,
                self::CODE_THREE => self::VALUE_THREE,
            ]
        );
    }

    /**
     * Sets custom attributes on a credit memo item using a GraphQl mutation and validates the saving of the attributes
     *
     * @param CreditmemoInterface $creditMemo
     * @param CreditmemoItemInterface $creditMemoItem
     * @param array $customAttributes
     * @param array $expectedAttributes
     * @return void
     * @throws \Magento\Framework\Exception\AuthenticationException
     */
    private function setCustomAttributesAndValidate(
        CreditmemoInterface $creditMemo,
        CreditmemoItemInterface $creditMemoItem,
        array $customAttributes,
        array $expectedAttributes
    ) {
        $response = $this->graphQlMutation(
            $this->queryHelper->getCustomAttributesOnCreditMemoItemMutationQuery(
                $this->uid->encode($creditMemo->getId()),
                $this->uid->encode($creditMemoItem->getId()),
                $customAttributes
            ),
            [],
            '',
            $this->customerAuthenticationHeader->execute()
        );

        self::assertArrayHasKey('setCustomAttributesOnCreditMemoItem', $response);
        self::assertArrayHasKey('credit_memo', $response['setCustomAttributesOnCreditMemoItem']);
        $responseCreditMemo = $response['setCustomAttributesOnCreditMemoItem']['credit_memo'];

        self::assertArrayHasKey('items', $responseCreditMemo);

        $responseCreditMemoItem = array_shift($responseCreditMemo['items']);
        self::assertArrayHasKey('custom_attributes', $responseCreditMemoItem);
        self::assertCount(count($expectedAttributes), $responseCreditMemoItem['custom_attributes']);

        $creditMemoItemRepository = $this->objectManager->get(CreditmemoItemRepositoryInterface::class);
        $creditMemoItem = $creditMemoItemRepository->get($creditMemoItem->getId());

        self::assertEquals(
            $expectedAttributes,
            $this->customAttributeConverter->toSerializableFormat($creditMemoItem->getCustomAttributes())
        );
    }
}
