<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\CustomAttributeSerializableGraphQl\Model\Resolver\Order;

use Magento\CustomAttributeSerializableGraphQl\Model\CustomAttributesToOutputConverter;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\Resolver\Value;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Sales\Api\OrderRepositoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Resolver for custom attributes field of order
 */
class CustomAttributes implements ResolverInterface
{
    /**
     * @param CustomAttributesToOutputConverter $customAttributesToOutputConverter
     * @param OrderRepositoryInterface $orderRepository
     * @param LoggerInterface $logger
     * @param Uid $uidEncoder
     */
    public function __construct(
        private readonly CustomAttributesToOutputConverter $customAttributesToOutputConverter,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly LoggerInterface $logger,
        private readonly Uid $uidEncoder
    ) {
    }

    /**
     * Return information about custom attributes for order
     *
     * @param Field            $field
     * @param ContextInterface $context
     * @param ResolveInfo      $info
     * @param array|null       $value
     * @param array|null       $args
     *
     * @return array|Value|mixed
     * @throws GraphQlInputException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        if (!isset($value['id'])) {
            throw new GraphQlInputException(__('"id" value should be specified'));
        }

        try {
            $orderId = $this->uidEncoder->isValidBase64($value['id']) ?
                $this->uidEncoder->decode($value['id']): $value['id'];

            $order = $this->orderRepository->get($orderId);

            return $this->customAttributesToOutputConverter->convert($order->getCustomAttributes());
        } catch (LocalizedException $e) {
            $this->logger->error(__('Can\'t load custom attributes for the order with id %1', $value['id']));

            return null;
        }
    }
}
