<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\ConfigurableProductDataExporter\Model\Provider\Product;

use Magento\DataExporter\Exception\UnableRetrieveData;
use Magento\Framework\App\ResourceConnection;
use Magento\ConfigurableProductDataExporter\Model\Query\ProductVariantQuery;
use Psr\Log\LoggerInterface;

class Variants
{
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var ProductVariantQuery
     */
    private $productVariantQuery;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Variants constructor.
     * @param ResourceConnection $resourceConnection
     * @param ProductVariantQuery $productVariantQuery
     * @param LoggerInterface $logger
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        ProductVariantQuery $productVariantQuery,
        LoggerInterface $logger
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->productVariantQuery = $productVariantQuery;
        $this->logger = $logger;
    }

    /**
     * @param array $row
     * @return array
     */
    private function format(array $row) : array
    {
        $output = $row;
        $output['variants'] = [
            'sku' => $row['sku'],
            'minimumPrice' => [
                'regularPrice' => $row['price'],
                'finalPrice' => $row['finalPrice']
            ],
            'selections' => $this->extractSingleValueAttributes($row)
        ];

        return $output;
    }

    /**
     * @param array $row
     * @return array
     */
    private function extractSingleValueAttributes(array $row) : array
    {
        $names = explode(',', $row['attributeNames']);
        $values = explode(',', $row['attributeValues']);
        $singleValueAttributes = [];
        foreach ($names as $key => $name) {
            $singleValueAttributes[] = [
                'name' => $name,
                'value' => $values[$key]
            ];
        }
        return $singleValueAttributes;
    }

    /**
     * @param array $values
     * @return array
     * @throws UnableRetrieveData
     */
    public function get(array $values) : array
    {
        $connection = $this->resourceConnection->getConnection();
        $queryArguments = [];
        try {
            $output = [];
            foreach ($values as $value) {
                $queryArguments['productId'][$value['productId']] = $value['productId'];
                $queryArguments['storeViewCode'][$value['storeViewCode']] = $value['storeViewCode'];
            }
            $select = $this->productVariantQuery->getQuery($queryArguments);
            $cursor = $connection->query($select);
            while ($row = $cursor->fetch()) {
                $output[] = $this->format($row);
            }
        } catch (\Exception $exception ) {
            $this->logger->error($exception->getMessage());
            throw new UnableRetrieveData(__('Unable to retrieve product variant data'));
        }
        return $output;
    }
}
