<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

namespace Magento\Checkout\Test\Unit\Block\Cart;

use Magento\Framework\TestFramework\Unit\Helper\ObjectManager as ObjectManagerHelper;

/**
 * Test for Magento\Checkout\Block\Cart\Grid.
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class GridTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var \Magento\Checkout\Block\Cart\Grid
     */
    private $block;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $itemCollectionFactoryMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $joinAttributeProcessorMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $scopeConfigMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $checkoutSessionMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $itemCollectionMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $quoteMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $layoutMock;

    /**
     * @var \PHPUnit_Framework_MockObject_MockObject
     */
    private $pagerBlockMock;

    protected function setUp()
    {
        $objectManagerHelper = new ObjectManagerHelper($this);
        $this->itemCollectionFactoryMock =
            $this->getMockBuilder(\Magento\Quote\Model\ResourceModel\Quote\Item\CollectionFactory::class)
                ->disableOriginalConstructor()
                ->setMethods(['create'])
                ->getMock();
        $this->joinAttributeProcessorMock =
            $this->getMockBuilder(\Magento\Framework\Api\ExtensionAttribute\JoinProcessorInterface::class)
                ->getMockForAbstractClass();
        $this->scopeConfigMock = $this->getMockBuilder(\Magento\Framework\App\Config\ScopeConfigInterface::class)
            ->getMockForAbstractClass();
        $this->checkoutSessionMock = $this->getMockBuilder(\Magento\Checkout\Model\Session::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->itemCollectionMock = $objectManagerHelper
            ->getCollectionMock(\Magento\Quote\Model\ResourceModel\Quote\Item\Collection::class, []);
        $this->quoteMock = $this->getMockBuilder(\Magento\Quote\Model\Quote::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->layoutMock = $this->getMockBuilder(\Magento\Framework\View\LayoutInterface::class)
            ->getMockForAbstractClass();
        $this->pagerBlockMock = $this->getMockBuilder(\Magento\Theme\Block\Html\Pager::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->checkoutSessionMock->expects($this->any())->method('getQuote')->willReturn($this->quoteMock);
        $this->quoteMock->expects($this->any())->method('getAllVisibleItems')->willReturn([]);
        $this->scopeConfigMock->expects($this->at(0))
            ->method('getValue')
            ->with(
                \Magento\Checkout\Block\Cart\Grid::XPATH_CONFIG_NUMBER_ITEMS_TO_DISPLAY_PAGER,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                null
            )->willReturn(20);
        $this->block = $objectManagerHelper->getObject(
            \Magento\Checkout\Block\Cart\Grid::class,
            [
                'itemCollectionFactory' => $this->itemCollectionFactoryMock,
                'joinAttributeProcessor' => $this->joinAttributeProcessorMock,
                'scopeConfig' => $this->scopeConfigMock,
                'checkoutSession' => $this->checkoutSessionMock,
                'layout' => $this->layoutMock,
                'data' => ['template' => 'cart/form1.phtml'],
            ]
        );
    }

    /**
     * @return void
     */
    public function testGetTemplate()
    {
        $this->assertEquals('cart/form1.phtml', $this->block->getTemplate());
    }

    /**
     * @covers \Magento\Checkout\Block\Cart\Grid::getItemsForGrid
     *
     * @return void
     */
    public function testGetItemsForGrid()
    {
        $this->getMockItemsForGrid();
        $this->assertEquals($this->itemCollectionMock, $this->block->getItemsForGrid());
    }

    /**
     * @cover \Magento\Checkout\Block\Cart\Grid::_prepareLayout
     *
     * @return void
     */
    public function testSetLayout()
    {
        $itemsCount = 150;
        $availableLimit = 20;
        $this->getMockItemsForGrid();
        $this->quoteMock->expects($this->once())->method('getItemsCount')->willReturn($itemsCount);
        $this->scopeConfigMock->expects($this->at(1))
            ->method('getValue')
            ->with(
                \Magento\Checkout\Block\Cart\Grid::XPATH_CONFIG_NUMBER_ITEMS_TO_DISPLAY_PAGER,
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                null
            )->willReturn($availableLimit);
        $this->layoutMock
            ->expects($this->once())
            ->method('createBlock')
            ->with(\Magento\Theme\Block\Html\Pager::class)
            ->willReturn($this->pagerBlockMock);
        $this->pagerBlockMock
            ->expects($this->once())
            ->method('setAvailableLimit')
            ->with([$availableLimit => $availableLimit])
            ->willReturnSelf();
        $this->pagerBlockMock
            ->expects($this->once())
            ->method('setCollection')
            ->with($this->itemCollectionMock)
            ->willReturnSelf();
        $this->layoutMock->expects($this->once())->method('setChild')->with(null, null, 'pager');
        $this->itemCollectionMock->expects($this->once())->method('load')->willReturnSelf();
        $this->quoteMock->expects($this->never())->method('getAllVisibleItems');
        $this->itemCollectionMock->expects($this->once())->method('getItems')->willReturn([]);
        $this->block->setLayout($this->layoutMock);
    }

    /**
     * @covers \Magento\Checkout\Block\Cart\Grid::getItems
     *
     * @return void
     */
    public function testGetItems()
    {
        $this->getMockItemsForGrid();
        $this->quoteMock->expects($this->once())->method('getItemsCount')->willReturn(20);
        $this->itemCollectionMock->expects($this->once())->method('getItems')->willReturn(['expected']);
        
        $this->assertEquals(['expected'], $this->block->getItems());
    }

    /**
     * Prepare data for getItemsForGrid() method.
     *
     * @return void
     */
    private function getMockItemsForGrid()
    {
        $this->itemCollectionFactoryMock
            ->expects($this->once())
            ->method('create')
            ->willReturn($this->itemCollectionMock);
        $this->checkoutSessionMock->expects($this->any())->method('getQuote')->willReturn($this->quoteMock);
        $this->itemCollectionMock->expects($this->once())->method('setQuote')->with($this->quoteMock)->willReturnSelf();
        $this->itemCollectionMock
            ->expects($this->once())
            ->method('addFieldToFilter')
            ->with('parent_item_id', ['null' => true])
            ->willReturnSelf();
        $this->joinAttributeProcessorMock->expects($this->once())->method('process')->with($this->itemCollectionMock);
    }

    /**
     * @cover \Magento\Checkout\Block\Cart::prepareItemUrls
     *
     * @return void
     */
    public function testGetItemsIfCustomItemsExists()
    {
        $itemMock = $this->getMockBuilder(\Magento\Quote\Model\Quote\Item::class)
            ->disableOriginalConstructor()
            ->getMock();
        $storeManager = $this->getMockBuilder(\Magento\Store\Model\StoreManagerInterface::class)
            ->getMockForAbstractClass();
        $storeMock = $this->getMockBuilder(\Magento\Store\Api\Data\StoreInterface::class)
            ->getMockForAbstractClass();
        $storeManager->expects($this->once())->method('getStore')->willReturn($storeMock);
        $objectManagerHelper = new ObjectManagerHelper($this);
        
        $this->block = $objectManagerHelper->getObject(
            \Magento\Checkout\Block\Cart\Grid::class,
            [
                'itemCollectionFactory' => $this->itemCollectionFactoryMock,
                'joinAttributeProcessor' => $this->joinAttributeProcessorMock,
                'scopeConfig' => $this->scopeConfigMock,
                'checkoutSession' => $this->checkoutSessionMock,
                'layout' => $this->layoutMock,
                'data' => ['custom_items' => [$itemMock]],
                'storeManager' => $storeManager,
            ]
        );
        
        $this->assertEquals([$itemMock], $this->block->getItems());
    }

    /**
     * Test getItems() method when pager is not displayed on page.
     *
     * @return void
     */
    public function testGetItemsWhenPagerNotVisible()
    {
        $this->assertEquals([], $this->block->getItems());
    }
}
