<?php
/**
 * Copyright 2019 Adobe
 * All Rights Reserved.
 */
declare(strict_types=1);

namespace Magento\CatalogUrlRewrite\Model;

use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Value as ConfigValue;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewrite;

/**
 * Table Cleaner in case of switching generate_category_product_rewrites off
 */
class TableCleaner extends ConfigValue
{
    const AUTO_GENERATED_ROW_FLAG = 1;
    const URL_REWRITE_GENERATION_OFF_FLAG = 0;

    /**
     * @var UrlRewrite
     */
    private $urlRewrite;

    /**
     * @param UrlRewrite $urlRewrite
     * @param Context $context
     * @param Registry $registry
     * @param ScopeConfigInterface $config
     * @param TypeListInterface $cacheTypeList
     * @param AbstractResource|null $resource
     * @param AbstractDb|null $resourceCollection
     * @param array $data
     */
    public function __construct(
        UrlRewrite $urlRewrite,
        Context $context,
        Registry $registry,
        ScopeConfigInterface $config,
        TypeListInterface $cacheTypeList,
        ?AbstractResource $resource = null,
        ?AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct($context, $registry, $config, $cacheTypeList, $resource, $resourceCollection, $data);
        $this->urlRewrite = $urlRewrite;
    }

    /**
     * @inheritDoc
     * @return ConfigValue
     * @throws LocalizedException
     */
    public function afterSave()
    {
        if ($this->getValue() == self::URL_REWRITE_GENERATION_OFF_FLAG) {
            $this->clearOldData();
        }
        return parent::afterSave();
    }

    /**
     * Clear urlrewrites for products in categories
     */
    private function clearOldData(): void
    {
        $tableName = $this->urlRewrite->getMainTable();
        $conditions = [
            'metadata LIKE ?' => '{"category_id"%',
            'is_autogenerated = ?' => self::AUTO_GENERATED_ROW_FLAG
        ];
        $this->urlRewrite->getConnection()->delete($tableName, $conditions);
    }
}
