<?php
/**
 * Copyright © 2013-2017 Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\CatalogUrlRewrite\Test\Unit\Model\Map;

use Magento\Framework\TestFramework\Unit\Helper\ObjectManager;
use Magento\CatalogUrlRewrite\Model\Map\UrlRewriteFinder;
use Magento\CatalogUrlRewrite\Model\Map\DatabaseMapPool;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Magento\CatalogUrlRewrite\Model\Map\DataCategoryUrlRewriteDatabaseMap;
use Magento\CatalogUrlRewrite\Model\Map\DataProductUrlRewriteDatabaseMap;

/**
 * Tests UrlRewriteFinder class.
 */
class UrlRewriteFinderTest extends \PHPUnit_Framework_TestCase
{
    /** @var DatabaseMapPool|\PHPUnit_Framework_MockObject_MockObject */
    private $databaseMapPoolMock;

    /** @var UrlRewriteFactory|\PHPUnit_Framework_MockObject_MockObject */
    private $urlRewriteFactoryMock;

    /** @var UrlRewrite|\PHPUnit_Framework_MockObject_MockObject */
    private $urlRewritePrototypeMock;

    /** @var UrlFinderInterface|\PHPUnit_Framework_MockObject_MockObject */
    private $urlFinderMock;

    /** @var UrlRewriteFinder|\PHPUnit_Framework_MockObject_MockObject */
    private $model;

    protected function setUp()
    {
        $this->databaseMapPoolMock = $this->getMock(DatabaseMapPool::class, [], [], '', false);
        $this->urlFinderMock = $this->getMock(UrlFinderInterface::class);
        $this->urlRewriteFactoryMock = $this->getMock(UrlRewriteFactory::class, ['create'], [], '', false);
        $this->urlRewritePrototypeMock = new UrlRewrite();

        $this->urlRewriteFactoryMock->expects($this->any())
            ->method('create')
            ->willReturn($this->urlRewritePrototypeMock);

        $urlRewriteClassesNamesArray = [
            UrlRewriteFinder::ENTITY_TYPE_PRODUCT => DataProductUrlRewriteDatabaseMap::class,
            UrlRewriteFinder::ENTITY_TYPE_CATEGORY => DataCategoryUrlRewriteDatabaseMap::class
        ];

        $this->model = (new ObjectManager($this))->getObject(
            UrlRewriteFinder::class,
            [
                'databaseMapPool' => $this->databaseMapPoolMock,
                'urlFinder' => $this->urlFinderMock,
                'urlRewriteFactory' => $this->urlRewriteFactoryMock,
                'urlRewriteClassNames' => $urlRewriteClassesNamesArray
            ]
        );
    }

    /**
     * Covers findAllByData() using urlFinder.
     *
     * @return void
     */
    public function testGetByIdentifiersFallback()
    {
        $expected = [1, 2, 3];
        $this->databaseMapPoolMock->expects($this->never())
            ->method('getDataMap');

        $this->urlFinderMock->expects($this->exactly(7))
            ->method('findAllByData')
            ->willReturn($expected);

        $this->assertEquals($expected, $this->model->findAllByData(1, 1, UrlRewriteFinder::ENTITY_TYPE_CATEGORY));
        $this->assertEquals($expected, $this->model->findAllByData(1, 1, UrlRewriteFinder::ENTITY_TYPE_PRODUCT));
        $this->assertEquals($expected, $this->model->findAllByData('a', 1, UrlRewriteFinder::ENTITY_TYPE_PRODUCT), 1);
        $this->assertEquals($expected, $this->model->findAllByData('a', 'a', UrlRewriteFinder::ENTITY_TYPE_PRODUCT), 1);
        $this->assertEquals($expected, $this->model->findAllByData(1, 'a', UrlRewriteFinder::ENTITY_TYPE_PRODUCT), 1);
        $this->assertEquals($expected, $this->model->findAllByData(1, 1, 'cms', 1));
        $this->assertEquals($expected, $this->model->findAllByData(1, 1, 'cms'));
    }

    /**
     * Covers findAllByData() Product URL rewrites.
     *
     * @return void
     */
    public function testGetByIdentifiersProduct()
    {
        $data =[
            [
                'url_rewrite_id' => '1',
                'entity_type' => 'product',
                'entity_id' => '3',
                'request_path' => 'request_path',
                'target_path' => 'target_path',
                'redirect_type' => 'redirect_type',
                'store_id' => '4',
                'description' => 'description',
                'is_autogenerated' => '1',
                'metadata' => '{}'
            ]
        ];

        $dataProductMapMock = $this->getMock(DataProductUrlRewriteDatabaseMap::class, [], [], '', false);
        $this->databaseMapPoolMock->expects($this->once())
            ->method('getDataMap')
            ->with(DataProductUrlRewriteDatabaseMap::class, 1)
            ->willReturn($dataProductMapMock);

        $this->urlFinderMock->expects($this->never())
            ->method('findAllByData')
            ->willReturn([]);

        $dataProductMapMock->expects($this->once())
            ->method('getData')
            ->willReturn($data);

        $urlRewriteResultArray = $this->model->findAllByData(1, 1, UrlRewriteFinder::ENTITY_TYPE_PRODUCT, 1);
        $this->assertEquals($data[0], $urlRewriteResultArray[0]->toArray());
    }

    /**
     * Covers findAllByData() Category URL rewrites.
     *
     * @return void
     */
    public function testGetByIdentifiersCategory()
    {
        $data =[
            [
                'url_rewrite_id' => '1',
                'entity_type' => 'category',
                'entity_id' => '3',
                'request_path' => 'request_path',
                'target_path' => 'target_path',
                'redirect_type' => 'redirect_type',
                'store_id' => '4',
                'description' => 'description',
                'is_autogenerated' => '1',
                'metadata' => '{}'
            ]
        ];

        $dataCategoryMapMock = $this->getMock(DataCategoryUrlRewriteDatabaseMap::class, [], [], '', false);
        $this->databaseMapPoolMock->expects($this->once())
            ->method('getDataMap')
            ->with(DataCategoryUrlRewriteDatabaseMap::class, 1)
            ->willReturn($dataCategoryMapMock);

        $this->urlFinderMock->expects($this->never())
            ->method('findAllByData')
            ->willReturn([]);

        $dataCategoryMapMock->expects($this->once())
            ->method('getData')
            ->willReturn($data);

        $urlRewriteResultArray = $this->model->findAllByData(1, 1, UrlRewriteFinder::ENTITY_TYPE_CATEGORY, 1);
        $this->assertEquals($data[0], $urlRewriteResultArray[0]->toArray());
    }
}
