<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\CatalogDataExporter\Test\Integration;

use Magento\TestFramework\Helper\Bootstrap;

/**
    * @magentoDataFixture loadFixture
    * @magentoDbIsolation disabled
    * @magentoAppIsolation enabled
    */
class ProductBuyableTest extends AbstractProductTestHelper
{

    /**
     * Load fixtures for test
     */
    public static function loadFixture()
    {
        include __DIR__ . '/_files/setup_buyable_product.php';
    }

    /**
     * Remove fixtures
     */
    public static function tearDownAfterClass()
    {
        include __DIR__ . '/_files/setup_buyable_product_rollback.php';
    }

    /**
     * @return array
     * @throws \Zend_Db_Statement_Exception
     */
    protected function setupExtractedProduct() : array
    {
        $sku = 'simple7';
        $storeViewCode = 'default';
        $extractedProduct = $this->getExtractedProduct($sku, $storeViewCode);
        return $extractedProduct;
    }

    /**
     * Validate buyable status
     *
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Zend_Db_Statement_Exception
     */
    public function testBuyableProduct() : void
    {
        $this->runIndexer();
        $this->changeBuyableStatus();
        $this->runIndexer();
        $this->validateProductBuyable($this->setupExtractedProduct());
    }

    /**
     * Validate buyable status
     *
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Zend_Db_Statement_Exception
     */
    public function testDisabledProduct() : void
    {
        $this->runIndexer();
        $this->disableProduct();
        $this->runIndexer();
        $this->validateDisabledProduct($this->setupExtractedProduct());
    }

    /**
     * Validate buyable status
     *
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Zend_Db_Statement_Exception
     */
    public function testEnabledProduct() : void
    {
        $this->runIndexer();
        $this->enableProduct();
        $this->runIndexer();
        $this->validateEnabledProduct($this->setupExtractedProduct());
    }

    /**
     * Change buyable status of product
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Exception
     */
    protected function changeBuyableStatus() : void
    {
        /** @var \Magento\Catalog\Api\ProductRepositoryInterface $productInterface */
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        $productInterface = $objectManager->create(\Magento\Catalog\Api\ProductRepositoryInterface::class);
        $product = $productInterface->get('simple7');
        $productId = $product->getId();

        /** @var \Magento\CatalogInventory\Model\Stock\Item $stockItem */
        $stockItem = Bootstrap::getObjectManager()->create(\Magento\CatalogInventory\Model\Stock\Item::class);
        $stockItem->load($productId, 'product_id');
        $stockItem->setIsInStock(false);
        $stockItem->save();
    }

    /**
     * Change buyable status of product
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Exception
     */
    protected function disableProduct() : void
    {
        /** @var \Magento\Catalog\Api\ProductRepositoryInterface $productInterface */
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $productInterface = $objectManager->create(\Magento\Catalog\Api\ProductRepositoryInterface::class);
        $product = $productInterface->get('simple7');
        $product->setStatus(\Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_DISABLED);
        $this->productRepository->save($product);
    }

    /**
     * Change buyable status of product
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws \Exception
     */
    protected function enableProduct() : void
    {
        /** @var \Magento\Catalog\Api\ProductRepositoryInterface $productInterface */
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $productInterface = $objectManager->create(\Magento\Catalog\Api\ProductRepositoryInterface::class);
        $product = $productInterface->get('simple7');
        $product->setStatus(\Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);
        $this->productRepository->save($product);

        $productId = $product->getId();
        /** @var \Magento\CatalogInventory\Model\Stock\Item $stockItem */
        $stockItem = Bootstrap::getObjectManager()->create(\Magento\CatalogInventory\Model\Stock\Item::class);
        $stockItem->load($productId, 'product_id');
        $stockItem->setIsInStock(true);
        $stockItem->save();
    }

    /**
     * Validate buyable status of product in extracted product data
     *
     * @param array $extractedProduct
     * @return void
     */
    protected function validateProductBuyable(array $extractedProduct) : void
    {
        $this->assertEquals(false, $extractedProduct['feedData']['buyable']);
    }

    /**
     * Validate buyable status of disabled product in extracted product data
     *
     * @param array $extractedProduct
     * @return void
     */
    protected function validateDisabledProduct(array $extractedProduct) : void
    {
        $this->assertEquals(false, $extractedProduct['feedData']['buyable']);
    }

    /**
     * Validate buyable status of enabled product in extracted product data
     *
     * @param array $extractedProduct
     * @return void
     */
    protected function validateEnabledProduct(array $extractedProduct) : void
    {
        $this->assertEquals(true, $extractedProduct['feedData']['buyable']);
    }
}
