<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\CatalogDataExporter\Model\Indexer;

use Magento\CatalogDataExporter\Model\Query\DeletedProductsQuery;
use Magento\DataExporter\Export\Processor;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Indexer\ActionInterface as IndexerActionInterface;
use Magento\Framework\Mview\ActionInterface as MviewActionInterface;
use Magento\Framework\Serialize\SerializerInterface;

/**
 * Class ProductFeedIndexer
 */
class ProductFeedIndexer implements IndexerActionInterface, MviewActionInterface
{
    /**
     * @var Processor
     */
    private $processor;

    /**
     * @var DeletedProductsQuery
     */
    private $deletedProductsQuery;

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var int
     */
    private $chunkSize;
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * ProductFeedIndexer constructor.
     *
     * @param Processor $processor
     * @param ResourceConnection $resourceConnection
     * @param DeletedProductsQuery $deletedProductsQuery
     * @param SerializerInterface $serializer
     * @param int $chunkSize
     */
    public function __construct(
        Processor $processor,
        ResourceConnection $resourceConnection,
        SerializerInterface $serializer,
        DeletedProductsQuery $deletedProductsQuery,
        int $chunkSize = 10
    ) {
        $this->processor = $processor;
        $this->resourceConnection = $resourceConnection;
        $this->deletedProductsQuery = $deletedProductsQuery;
        $this->chunkSize = $chunkSize;
        $this->serializer = $serializer;
    }

    /**
     * Execute full indexation
     *
     * @return void
     */
    public function executeFull()
    {
        $this->process();
    }

    /**
     * Execute partial indexation by ID list
     *
     * @param int[] $ids
     * @return void
     */
    public function executeList(array $ids)
    {
        $this->process($ids);
    }

    /**
     * Execute partial indexation by ID
     *
     * @param int $id
     * @return void
     */
    public function executeRow($id)
    {
        $this->process([$id]);
    }

    /**
     * Execute materialization on ids entities
     *
     * @param int[] $ids
     * @return void
     * @api
     */
    public function execute($ids)
    {
        $this->process($ids);
    }

    /**
     * @param array $data
     * @return array
     */
    private function formatData(array $data): array
    {
        $output = [];
        foreach ($data as $row) {
            $output[] = [
                'sku' => $row['sku'],
                'store_view_code' => $row['storeViewCode'],
                'feed_data' => $this->serializer->serialize($row)
            ];
        }
        return $output;
    }

    /**
     * @param array $ids
     */
    private function process($ids = [])
    {
        $data = $this->processor->process('products', $ids);
        $chunks = array_chunk($data, $this->chunkSize);
        $connection = $this->resourceConnection->getConnection();
        $this->deletedProductsQuery->updateDeletedFlagQuery();
        foreach ($chunks as $chunk) {
            $connection->insertOnDuplicate(
                $this->resourceConnection->getTableName('catalog_data_exporter_products'),
                $this->formatData($chunk),
                ['feed_data', 'is_deleted']
            );
        }
    }
}
