<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\CatalogDataExporter\Model\Feed;

use Magento\DataExporter\Model\FeedInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Serialize\SerializerInterface;

class Products implements FeedInterface
{
    /**
     * @var ResourceConnection
     */
    private $resourceConnection;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * Products constructor.
     *
     * @param ResourceConnection $resourceConnection
     * @param SerializerInterface $serializer
     */
    public function __construct(
        ResourceConnection $resourceConnection,
        SerializerInterface $serializer
    ) {
        $this->resourceConnection = $resourceConnection;
        $this->serializer = $serializer;
    }

    /**
     * @param string $timestamp
     * @return array
     * @throws \Zend_Db_Statement_Exception
     */
    public function getFeedSince(string $timestamp): array
    {
        $connection = $this->resourceConnection->getConnection();
        $recentTimestamp = null;
        $select = $connection->select()
            ->from(
                ['t' => $this->resourceConnection->getTableName('catalog_data_exporter_products')],
                [
                    'feed_data',
                    'modified_at',
                    'is_deleted'
                ]
            )
            ->where('t.modified_at > ?', $timestamp);
        $cursor = $connection->query($select);
        $output = [];
        while ($row = $cursor->fetch()) {
            $dataRow = $this->serializer->unserialize($row['feed_data']);
            $dataRow['modifiedAt'] = $row['modified_at'];
            $dataRow['deleted'] = (bool) $row['is_deleted'];
            $output[] = $dataRow;
            if ($recentTimestamp == null || $recentTimestamp < $row['modified_at']) {
                $recentTimestamp = $row['modified_at'];
            }
        }
        return [
            'recentTimestamp' => $recentTimestamp,
            'feed' => $output
        ];
    }
}
