<?php

namespace Magento\AdobeIoEvents\Console;

use Magento\AdobeIoEvents\Model\AdobeIOConfigurationProvider;
use Magento\AdobeIoEvents\Model\EventMetadataRegistry;
use Magento\AdobeIoEvents\Model\IOEventsAPIClient;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\AuthorizationException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NotFoundException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class CreateEventProvider extends Command
{
    public const COMMAND_NAME = "events:create-event-provider";

    /**
     * @var AdobeIOConfigurationProvider
     */
    private AdobeIOConfigurationProvider $configurationProvider;

    /**
     * @var EventMetadataRegistry
     */
    private EventMetadataRegistry $eventMetadataRegistry;

    /**
     * @var IOEventsAPIClient
     */
    private IOEventsAPIClient $IOEventsAPIClient;

    /**
     * @param AdobeIOConfigurationProvider $configurationProvider
     * @param EventMetadataRegistry $eventMetadataRegistry
     * @param IOEventsAPIClient $IOEventsAPIClient
     */
    public function __construct(
        AdobeIOConfigurationProvider $configurationProvider,
        EventMetadataRegistry $eventMetadataRegistry,
        IOEventsAPIClient $IOEventsAPIClient
    ) {
        $this->configurationProvider = $configurationProvider;
        $this->eventMetadataRegistry = $eventMetadataRegistry;
        $this->IOEventsAPIClient = $IOEventsAPIClient;

        parent::__construct();
    }

    protected function configure()
    {
        $this->setName(self::COMMAND_NAME);
        $this->setDescription("Create a custom event provider in IO events for this instance.");

        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return void|null
     * @throws AlreadyExistsException
     * @throws AuthorizationException
     * @throws InputException
     * @throws AuthenticationException
     * @throws NotFoundException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $provider = $this->configurationProvider->retrieveProvider();
        $instanceId = $this->configurationProvider->retrieveInstanceId();

        if ($provider !== null) {
            $output->writeln("Already found an event provider configured with ID " . $provider->getId());
            return null;
        }

        $output->writeln("No event provider found, a new event provider will be created");

        $declaredProvider = $this->eventMetadataRegistry->getDeclaredEventProvider();

        $provider = $this->IOEventsAPIClient->createEventProvider(
            $instanceId,
            $declaredProvider
        );

        $this->configurationProvider->saveProvider($provider);
        $output->writeln("A new event provider has been created with ID " . $provider->getId());
    }
}
