<?php

namespace Magento\AdobeIoEvents\Model;

use GuzzleHttp\ClientFactory;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\ResponseFactory;
use Magento\AdobeIms\Model\OAuth\TokenResponse;
use Magento\AdobeIoEvents\Model\Data\AdobeConsoleConfiguration\AdobeConsoleConfiguration;
use Magento\AdobeIoEvents\Model\Data\EventMetadata;
use Magento\AdobeIoEvents\Model\Data\EventMetadataFactory;
use Magento\AdobeIoEvents\Model\Data\EventProvider;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Store\Model\ScopeInterface;

class CreateEventsMetadata
{
    public const XML_PATH_ADOBE_IO_EVENTS_CREATION_URL = 'adobe_io_events/integration/adobe_io_event_creation_url';
    public const XML_PATH_ADOBE_IO_EVENTS_TYPE_LIST_URL = 'adobe_io_events/integration/adobe_io_event_type_list_url';
    public const XML_PATH_ADOBE_IO_EVENTS_TYPE_DELETE_URL = 'adobe_io_events/integration/adobe_io_event_type_delete_url';

    /**
     * @var ResponseFactory
     */
    private ResponseFactory $responseFactory;

    /**
     * @var ClientFactory
     */
    private ClientFactory $clientFactory;

    /**
     * @var ScopeConfigInterface
     */
    private ScopeConfigInterface $scopeConfig;

    /**
     * @var Json
     */
    private Json $json;

    /**
     * @var EventMetadataFactory
     */
    private EventMetadataFactory $eventMetadataFactory;

    /**
     * @param ResponseFactory $responseFactory
     * @param ClientFactory $clientFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param Json $json
     * @param EventMetadataFactory $eventMetadataFactory
     */
    public function __construct(
        ResponseFactory $responseFactory,
        ClientFactory $clientFactory,
        ScopeConfigInterface $scopeConfig,
        Json $json,
        EventMetadataFactory $eventMetadataFactory
    ) {
        $this->responseFactory = $responseFactory;
        $this->clientFactory = $clientFactory;
        $this->scopeConfig = $scopeConfig;
        $this->json = $json;
        $this->eventMetadataFactory = $eventMetadataFactory;
    }

    /**
     * Adobe IO Events API to create Event Metadata
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @param TokenResponse $accessToken
     * @param EventMetadata $eventType
     * @return void
     * @throws GuzzleException
     */
    public function createEventMetadata(
        AdobeConsoleConfiguration $configuration,
        EventProvider $provider,
        TokenResponse $accessToken,
        EventMetadata $eventType
    ) {
        $credentials = $configuration->getFirstCredential();

        $uri = $this->getEventMetadataUri($configuration, $provider);

        $params = [
            'json' => $eventType->jsonSerialize()
        ];

        $this->doRequest(Request::HTTP_METHOD_POST, $uri, $credentials, $accessToken, $params);
    }

    /**
     * Adobe IO Events API to list Event Metadata for an Event Provider
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @param TokenResponse $accessToken
     * @return EventMetadata[]
     * @throws GuzzleException
     */
    public function listRegisteredEventMetadata(
        AdobeConsoleConfiguration $configuration,
        EventProvider $provider,
        TokenResponse $accessToken
    ): array {
        $credentials = $configuration->getFirstCredential();
        $uri = $this->getEventMetadataListUri($configuration, $provider);

        $response = $this->doRequest(Request::HTTP_METHOD_GET, $uri, $credentials, $accessToken);
        $data = $this->json->unserialize($response->getBody()->getContents());

        $eventTypeList = [];
        foreach ($data["_embedded"]["eventmetadata"] as $eventTypeData) {
            $eventType = $this->eventMetadataFactory->create();
            $eventType->setLabel($eventTypeData["label"]);
            $eventType->setDescription($eventTypeData["description"]);
            $eventType->setEventCode($eventTypeData["event_code"]);

            $eventTypeList[] = $eventType;
        }

        return $eventTypeList;
    }

    /**
     * Adobe IO Events API to delete an Event Metadata
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @param TokenResponse $accessToken
     * @param EventMetadata $eventType
     * @return bool
     * @throws GuzzleException
     */
    public function deleteEventMetadata(
        AdobeConsoleConfiguration $configuration,
        EventProvider $provider,
        TokenResponse $accessToken,
        EventMetadata $eventType
    ): bool {
        $credentials = $configuration->getFirstCredential();

        $uri = $this->getEventMetadataDeleteUri($configuration, $provider, $eventType->getEventCode());

        $response = $this->doRequest(Request::HTTP_METHOD_DELETE, $uri, $credentials, $accessToken);

        return $response->getStatusCode() == 204;
    }

    /**
     * Compute Event Metadata Delete URI
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @param string $eventCode
     * @return string
     */
    private function getEventMetadataDeleteUri(
        AdobeConsoleConfiguration $configuration,
        EventProvider $provider,
        string $eventCode
    ): string {
        return str_replace(
            ["#{ims_org_id}", "#{project_id}", "#{workspace_id}", "#{provider_id}", "#{event_code}"],
            [
                $configuration->getProject()->getOrganization()->getId(),
                $configuration->getProject()->getId(),
                $configuration->getProject()->getWorkspace()->getId(),
                $provider->getId(),
                $eventCode
            ],
            $this->scopeConfig->getValue(self::XML_PATH_ADOBE_IO_EVENTS_TYPE_DELETE_URL, ScopeInterface::SCOPE_STORE),
        );
    }

    /**
     * Compute Event Metadata List URI
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @return string
     */
    private function getEventMetadataListUri(AdobeConsoleConfiguration $configuration, EventProvider $provider): string
    {
        return str_replace(
            ["#{ims_org_id}", "#{project_id}", "#{workspace_id}", "#{provider_id}"],
            [
                $configuration->getProject()->getOrganization()->getId(),
                $configuration->getProject()->getId(),
                $configuration->getProject()->getWorkspace()->getId(),
                $provider->getId()
            ],
            $this->scopeConfig->getValue(self::XML_PATH_ADOBE_IO_EVENTS_TYPE_LIST_URL, ScopeInterface::SCOPE_STORE),
        );
    }

    /**
     * Compute Event Metadata Creation URI
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param EventProvider $provider
     * @return array|mixed|string|string[]
     */
    private function getEventMetadataUri(AdobeConsoleConfiguration $configuration, EventProvider $provider)
    {
        $uri = str_replace(
            ["#{ims_org_id}", "#{project_id}", "#{workspace_id}", "#{provider_id}"],
            [
                $configuration->getProject()->getOrganization()->getId(),
                $configuration->getProject()->getId(),
                $configuration->getProject()->getWorkspace()->getId(),
                $provider->getId()
            ],
            $this->scopeConfig->getValue(self::XML_PATH_ADOBE_IO_EVENTS_CREATION_URL, ScopeInterface::SCOPE_STORE),
        );
        return $uri;
    }

    /**
     * Performars HTTP Request to Adobe IO Events
     *
     * @param string $method
     * @param string $uri
     * @param Data\AdobeConsoleConfiguration\Credentials $credentials
     * @param TokenResponse $accessToken
     * @param array $params
     * @return Response
     */
    private function doRequest(
        string $method,
        string $uri,
        Data\AdobeConsoleConfiguration\Credentials $credentials,
        TokenResponse $accessToken,
        array $params = []
    ): Response {
        $client = $this->clientFactory->create();

        $params['headers'] =
            [
                'x-api-key' => $credentials->getJwt()->getClientId(),
                "Authorization" => "Bearer " . $accessToken->getAccessToken()
            ];

        try {
            $response =  $client->request($method, $uri, $params);
        } catch (GuzzleException $exception) {
            $response = $this->responseFactory->create([
                'status' => $exception->getCode(),
                'reason' => $exception->getMessage()
            ]);
        }

        return $response;
    }
}
