<?php

namespace Magento\AdobeIoEvents\Console;

use Magento\AdobeIoEvents\Model\Data\AdobeConsoleConfiguration\AdobeConsoleConfiguration;
use Magento\AdobeIoEvents\Model\Data\AdobeConsoleConfiguration\ConfigurationFactory;
use Magento\AdobeIoEvents\Model\Data\PrivateKey;
use Magento\AdobeIoEvents\Model\Data\PrivateKeyFactory;
use Magento\Framework\Filesystem\DriverPool;
use Magento\Framework\Filesystem\File\ReadFactory;
use Magento\Framework\Serialize\Serializer\Json;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Exception\MissingInputException;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;

class AbstractJwtAuthConsole extends Command
{
    public const ENV_ADOBE_IO_TECHACCT_PRIVATE_KEY = 'ADOBE_IO_TECHACCT_PRIVATE_KEY';
    public const ENV_ADOBE_IO_CONSOLE_CONFIGURATION = 'ADOBE_IO_CONSOLE_CONFIGURATION';

    /**
     * @var PrivateKeyFactory
     */
    private PrivateKeyFactory $privateKeyFactory;

    /**
     * @var ReadFactory
     */
    private ReadFactory $readFactory;

    /**
     * @var Json
     */
    private Json $json;

    /**
     * @var ConfigurationFactory
     */
    private ConfigurationFactory $configurationFactory;

    /**
     * @param PrivateKeyFactory $privateKeyFactory
     * @param ReadFactory $readFactory
     * @param Json $json
     * @param ConfigurationFactory $configurationFactory
     */
    public function __construct(
        PrivateKeyFactory $privateKeyFactory,
        ReadFactory $readFactory,
        Json $json,
        ConfigurationFactory $configurationFactory
    ) {
        $this->privateKeyFactory = $privateKeyFactory;
        $this->readFactory = $readFactory;
        $this->json = $json;
        $this->configurationFactory = $configurationFactory;

        parent::__construct();
    }

    protected function configure()
    {
        $this->addOption(
            'configuration',
            'c',
            InputOption::VALUE_REQUIRED,
            'Path to the workspace credential JSON file. It should be downloaded from Adobe Console.'
        );

        $this->addOption(
            'private-key',
            'k',
            InputOption::VALUE_REQUIRED,
            'Path to the technical account private key.'
        );

        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @return PrivateKey
     */
    protected function getPrivateKey(InputInterface $input): PrivateKey
    {
        $privateKeyData = getenv(self::ENV_ADOBE_IO_TECHACCT_PRIVATE_KEY);
        $privateKey = $this->privateKeyFactory->create();

        if (!$privateKeyData && !$input->getOption("private-key")) {
            throw new MissingInputException("Private Key missing from environment variable " . self::ENV_ADOBE_IO_TECHACCT_PRIVATE_KEY);
        }

        if (!$privateKeyData) {
            $file = $this->readFactory->create($input->getOption("private-key"), DriverPool::FILE);
            $privateKeyData = $file->readAll();
            $privateKey->setData($privateKeyData);
        }

        $privateKey->setData($privateKeyData);

        return $privateKey;
    }

    /**
     * @param InputInterface $input
     * @return AdobeConsoleConfiguration
     */
    protected function getConfiguration(InputInterface $input): AdobeConsoleConfiguration
    {
        $content = getenv(self::ENV_ADOBE_IO_CONSOLE_CONFIGURATION);

        if (!$content && !$input->getOption("configuration")) {
            throw new MissingInputException("Configuration missing from environment variable " . self::ENV_ADOBE_IO_CONSOLE_CONFIGURATION);
        }

        if (!$content) {
            $file = $this->readFactory->create($input->getOption("configuration"), DriverPool::FILE);
            $content = $file->readAll();
        }

        $data = $this->json->unserialize($content);
        return $this->configurationFactory->create($data);
    }
}
