<?php

namespace Magento\AdobeIoEvents\Model;

use GuzzleHttp\ClientFactory;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\ResponseFactory;
use Magento\AdobeIms\Model\OAuth\TokenResponse;
use Magento\AdobeIoEvents\Api\EventMetadataRegistryInterface;
use Magento\AdobeIoEvents\Model\Data\AdobeConsoleConfiguration\AdobeConsoleConfiguration;
use Magento\AdobeIoEvents\Model\Data\EventProvider;
use Magento\AdobeIoEvents\Model\Data\EventProviderFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\AlreadyExistsException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Phrase;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Webapi\Rest\Request;
use Symfony\Component\Console\Exception\MissingInputException;
use Symfony\Component\Console\Output\OutputInterface;

class CreateEventProvider
{
    private const XML_PATH_ADOBE_IO_PROVIDER_URL = 'adobe_io_events/integration/adobe_io_provider_url';
    private const ENV_ADOBE_IO_EVENTS_PROVIDER_ID = 'ADOBE_IO_EVENTS_PROVIDER_ID';
    private const ENV_ADOBE_IO_EVENTS_INSTANCE_ID = "ADOBE_IO_EVENTS_INSTANCE_ID";

    /**
     * @var ResponseFactory
     */
    private ResponseFactory $responseFactory;

    /**
     * @var ClientFactory
     */
    private ClientFactory $clientFactory;

    /**
     * @var ScopeConfigInterface
     */
    private ScopeConfigInterface $scopeConfig;

    /**
     * @var Json
     */
    private Json $json;

    /**
     * @var EventProviderFactory
     */
    private EventProviderFactory $eventProviderFactory;

    /**
     * @var EventMetadataRegistryInterface
     */
    private EventMetadataRegistryInterface $eventTypeRegistry;

    /**
     * @param ResponseFactory $responseFactory
     * @param ClientFactory $clientFactory
     * @param ScopeConfigInterface $scopeConfig
     * @param Json $json
     * @param EventProviderFactory $eventProviderFactory
     * @param EventMetadataRegistryInterface $eventTypeRegistry
     */
    public function __construct(
        ResponseFactory $responseFactory,
        ClientFactory $clientFactory,
        ScopeConfigInterface $scopeConfig,
        Json $json,
        EventProviderFactory $eventProviderFactory,
        EventMetadataRegistryInterface $eventTypeRegistry
    ) {
        $this->responseFactory = $responseFactory;
        $this->clientFactory = $clientFactory;
        $this->scopeConfig = $scopeConfig;
        $this->json = $json;
        $this->eventProviderFactory = $eventProviderFactory;
        $this->eventTypeRegistry = $eventTypeRegistry;
    }

    /**
     * Adobe IO Event API to create an Event Provider
     *
     * @param AdobeConsoleConfiguration $configuration
     * @param TokenResponse $accessToken
     * @param OutputInterface $output
     * @return EventProvider
     * @throws AlreadyExistsException
     * @throws InputException
     */
    public function execute(
        AdobeConsoleConfiguration $configuration,
        TokenResponse $accessToken,
        OutputInterface $output
    ): EventProvider {
        $provider = $this->retrieveProvider();
        $instanceId = $this->retrieveInstanceId();

        if ($provider !== null()) {
            $output->writeln("Already found an event provider configured with ID " . $provider->getId());
            return $provider;
        }

        $output->writeln("No event provider found, a new event provider will be created");

        $response = $this->doRequest($accessToken, $configuration, $instanceId);

        if ($response->getStatusCode() == 409) {
            throw new AlreadyExistsException(new Phrase("An event provider with the same instance ID already exists."));
        }

        if ($response->getStatusCode() != 201) {
            throw new InputException(new Phrase($response->getReasonPhrase()));
        }

        $body = $response->getBody()->getContents();
        $data = $this->json->unserialize($body);

        $provider = $this->eventProviderFactory->create(["data" => $data]);

        $output->writeln("A new event provider has been created. Please run the following command: ");
        $output->write(
            sprintf(
                "\tmagento-cloud variable:set env:%s '%s'\n",
                self::ENV_ADOBE_IO_EVENTS_PROVIDER_ID,
                $provider->getId()
            )
        );

        return $provider;
    }

    /**
     * Retrieve Event Provider ID
     *
     * @return EventProvider|null
     */
    public function retrieveProvider(): ?EventProvider
    {
        $providerId = getenv(self::ENV_ADOBE_IO_EVENTS_PROVIDER_ID);

        if ($providerId) {
            return $this->eventProviderFactory->create(['data' => ['id' => $providerId]]);
        }

        return null;
    }

    /**
     * Perform HTTP Request to Adobe IO Events
     *
     * @param TokenResponse $token
     * @param AdobeConsoleConfiguration $configuration
     * @param string $instanceId
     * @return Response
     */
    private function doRequest(
        TokenResponse $token,
        AdobeConsoleConfiguration $configuration,
        string $instanceId
    ): Response {
        $client = $this->clientFactory->create();
        $declaredProvider = $this->eventTypeRegistry->getDeclaredEventProvider();

        $payload = [
            //"provider_metadata" => "aep_magento_saas_events",
            "instance_id" => $instanceId,
            "label" => $declaredProvider->getLabel(),
            "description" => sprintf("%s (Instance %s)", $declaredProvider->getDescription(), $instanceId)
        ];

        $credentials = ($configuration->getProject()->getWorkspace()->getDetails()->getCredentials())[0];

        $uriEndpoint = str_replace(
            ["#{ims_org_id}", "#{project_id}", "#{workspace_id}"],
            [
                $configuration->getProject()->getOrganization()->getId(),
                $configuration->getProject()->getId(),
                $configuration->getProject()->getWorkspace()->getId()
            ],
            $this->scopeConfig->getValue(self::XML_PATH_ADOBE_IO_PROVIDER_URL)
        );

        $params['headers'] = [
            'x-api-key' => $credentials->getJwt()->getClientId(),
            'Authorization' => 'Bearer ' . $token->getAccessToken(),
        ];

        $params['json'] = $payload;

        try {
            $response = $client->request(
                Request::HTTP_METHOD_POST,
                $uriEndpoint,
                $params
            );
        } catch (GuzzleException $exception) {
            $response = $this->responseFactory->create([
                'status' => $exception->getCode(),
                'reason' => $exception->getMessage()
            ]);
        }

        return $response;
    }

    /**
     * Retrieve Instance ID
     *
     * @return array|string
     */
    private function retrieveInstanceId()
    {
        $instanceId = getenv(self::ENV_ADOBE_IO_EVENTS_INSTANCE_ID);

        if (!$instanceId || $instanceId == "") {
            throw new MissingInputException(
                "Instance Id missing from environment variable " . self::ENV_ADOBE_IO_EVENTS_INSTANCE_ID
            );
        }

        return $instanceId;
    }
}
