<?php

namespace Magento\AdobeIoEvents\Console;

use GuzzleHttp\Exception\GuzzleException;
use Magento\AdobeIoEvents\Api\EventMetadataRegistryInterface;
use Magento\AdobeIoEvents\Model\CreateEventsMetadata;
use Magento\AdobeIoEvents\Model\Data\AdobeConsoleConfiguration\ConfigurationFactory;
use Magento\AdobeIoEvents\Model\Data\EventMetadata;
use Magento\AdobeIoEvents\Model\Data\PrivateKeyFactory;
use Magento\AdobeIoEvents\Model\JwtAccessToken;
use Magento\Framework\Exception\AuthorizationException;
use Magento\Framework\Filesystem\File\ReadFactory;
use Magento\Framework\Serialize\Serializer\Json;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class SynchronizeEventMetadata extends AbstractJwtAuthConsole
{
    /**
     * @var CreateEventsMetadata
     */
    private CreateEventsMetadata $createEventTypes;

    /**
     * @var JwtAccessToken
     */
    private JwtAccessToken $jwtAccessToken;

    /**
     * @var \Magento\AdobeIoEvents\Model\CreateEventProvider
     */
    private \Magento\AdobeIoEvents\Model\CreateEventProvider $createProvider;

    /**
     * @var EventMetadataRegistryInterface
     */
    private EventMetadataRegistryInterface $eventRegistry;

    /**
     * @param CreateEventsMetadata $createEventTypes
     * @param JwtAccessToken $jwtAccessToken
     * @param \Magento\AdobeIoEvents\Model\CreateEventProvider $createProvider
     * @param EventMetadataRegistryInterface $eventRegistry
     * @param Json $json
     * @param ConfigurationFactory $configurationFactory
     * @param ReadFactory $readFactory
     * @param PrivateKeyFactory $privateKeyFactory
     */
    public function __construct(
        CreateEventsMetadata $createEventTypes,
        JwtAccessToken $jwtAccessToken,
        \Magento\AdobeIoEvents\Model\CreateEventProvider $createProvider,
        EventMetadataRegistryInterface $eventRegistry,
        Json $json,
        ConfigurationFactory $configurationFactory,
        ReadFactory $readFactory,
        PrivateKeyFactory $privateKeyFactory
    ) {
        $this->createEventTypes = $createEventTypes;
        $this->jwtAccessToken = $jwtAccessToken;
        $this->createProvider = $createProvider;
        $this->eventRegistry = $eventRegistry;

        parent::__construct($privateKeyFactory, $readFactory, $json, $configurationFactory);
    }

    protected function configure()
    {
        $this->setName("events:sync-events-metadata");
        $this->setDescription("Synchronise event metadata for this instance");

        $this->addOption(
            "delete",
            "d",
            InputOption::VALUE_NONE,
            "Delete events metadata no longer required"
        );

        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return void
     * @throws AuthorizationException
     * @throws GuzzleException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $privateKey = $this->getPrivateKey($input);
        $configuration = $this->getConfiguration($input);

        $token = $this->jwtAccessToken->getAccessToken($configuration, $privateKey);

        $provider = $this->createProvider->retrieveProvider();
        if (is_null($provider)) {
            $output->writeln(
                sprintf(
                    "<error>No event provider is configured, please run bin/magento %s</error>",
                    CreateEventProvider::COMMAND_NAME
                )
            );
            return;
        }

        $output->writeln(
            sprintf(
                "Event provider with ID <info>%s</info> retrieved from configuration",
                $provider->getId()
            )
        );

        $output->writeln("<info>The following events are declared on your instance:</info>");
        $declaredEventMetadata = $this->getEventMetadataList();
        foreach ($declaredEventMetadata as $eventMetadata) {
            $output->writeln("- $eventMetadata");
        }

        $registeredEventMetadata = $this->createEventTypes->listRegisteredEventMetadata($configuration, $provider, $token);

        $eventTypeToDelete = array_diff($registeredEventMetadata, $declaredEventMetadata);

        $output->writeln("<info>Updating event types:</info>");
        foreach ($declaredEventMetadata as $eventType) {
            $this->createEventTypes->createEventMetadata($configuration, $provider, $token, $eventType);
            $output->writeln("- <info>[UPDATED]</info> $eventType");
        }

        if (count($eventTypeToDelete) > 0) {
            if ($input->getOption("delete")) {
                $output->writeln("<info>Delete the following event metedata:</info>");
                foreach ($eventTypeToDelete as $eventType) {
                    $deleted = $this->createEventTypes->deleteEventMetadata($configuration, $provider, $token, $eventType);
                    if ($deleted) {
                        $output->writeln("- <comment>[DELETED]</comment> $eventType");
                    } else {
                        $output->writeln("- <error>[FAILURE]</error> $eventType");
                    }
                }
            } else {
                $output->writeln("<info>The following event metadata could be deleted, by using --delete option</info>");
                foreach ($eventTypeToDelete as $eventType) {
                    $output->writeln("- $eventType");
                }
            }
        }
    }

    /**
     * @return EventMetadata[]
     */
    private function getEventMetadataList(): array
    {
        return $this->eventRegistry->getDeclaredEventMetadataList();
    }
}
