<?php

declare(strict_types=1);

namespace Laminas\Crypt\PublicKey;

use Laminas\Crypt\Exception;
use Laminas\Crypt\Exception\InvalidArgumentException;
use Laminas\Crypt\Exception\RuntimeException;
use Laminas\Math;
use Laminas\Math\BigInteger\Adapter\AdapterInterface;

use function function_exists;
use function mb_strlen;
use function openssl_dh_compute_key;
use function openssl_error_string;
use function openssl_pkey_get_details;
use function openssl_pkey_new;
use function preg_match;

use const OPENSSL_KEYTYPE_DH;
use const PHP_VERSION_ID;

/**
 * PHP implementation of the Diffie-Hellman public key encryption algorithm.
 * Allows two unassociated parties to establish a joint shared secret key
 * to be used in encrypting subsequent communications.
 */
class DiffieHellman
{
    public const DEFAULT_KEY_SIZE = 2048;

    /**
     * Key formats
     */
    public const FORMAT_BINARY = 'binary';
    public const FORMAT_NUMBER = 'number';
    public const FORMAT_BTWOC  = 'btwoc';

    /**
     * Static flag to select whether to use PHP5.3's openssl extension
     * if available.
     *
     * @var bool
     */
    public static $useOpenssl = true;

    /**
     * Default large prime number; required by the algorithm.
     */
    private string $prime;

    /**
     * The default generator number. This number must be greater than 0 but
     * less than the prime number set.
     */
    private string $generator;

    /**
     * A private number set by the local user. It's optional and will
     * be generated if not set.
     */
    private ?string $privateKey = null;

    /**
     * BigInteger support object courtesy of Laminas\Math
     */
    private readonly AdapterInterface $math;

    /**
     * The public key generated by this instance after calling generateKeys().
     */
    private ?string $publicKey = null;

    /**
     * The shared secret key resulting from a completed Diffie Hellman
     * exchange
     */
    private ?string $secretKey = null;

    /** @var resource */
    protected $opensslKeyResource;

    /**
     * Constructor; if set construct the object using the parameter array to
     * set values for Prime, Generator and Private.
     * If a Private Key is not set, one will be generated at random.
     */
    public function __construct(
        string $prime,
        string $generator,
        ?string $privateKey = null,
        string $privateKeyFormat = self::FORMAT_NUMBER
    ) {
        // set up BigInteger adapter
        $this->math = Math\BigInteger\BigInteger::factory();

        $this->setPrime($prime);
        $this->setGenerator($generator);
        if ($privateKey !== null) {
            $this->setPrivateKey($privateKey, $privateKeyFormat);
        }
    }

    /**
     * Set whether to use openssl extension
     *
     * @static
     */
    public static function useOpensslExtension(bool $flag = true): void
    {
        static::$useOpenssl = $flag;
    }

    /**
     * Generate own public key. If a private number has not already been set,
     * one will be generated at this stage.
     *
     * @throws RuntimeException
     */
    public function generateKeys(): static
    {
        if (function_exists('openssl_dh_compute_key') && static::$useOpenssl !== false) {
            $details = [
                'p' => $this->convert($this->getPrime(), self::FORMAT_NUMBER, self::FORMAT_BINARY),
                'g' => $this->convert($this->getGenerator(), self::FORMAT_NUMBER, self::FORMAT_BINARY),
            ];
            // the priv_key parameter is allowed only for PHP < 7.1
            // @see https://bugs.php.net/bug.php?id=73478
            if ($this->hasPrivateKey() && PHP_VERSION_ID < 70100) {
                $details['priv_key'] = $this->convert(
                    $this->privateKey,
                    self::FORMAT_NUMBER,
                    self::FORMAT_BINARY
                );
                $opensslKeyResource  = openssl_pkey_new(['dh' => $details]);
            } else {
                $opensslKeyResource = openssl_pkey_new([
                    'dh'               => $details,
                    'private_key_bits' => self::DEFAULT_KEY_SIZE,
                    'private_key_type' => OPENSSL_KEYTYPE_DH,
                ]);
            }

            if (false === $opensslKeyResource) {
                throw new Exception\RuntimeException(
                    'Can not generate new key; openssl ' . openssl_error_string()
                );
            }

            $data = openssl_pkey_get_details($opensslKeyResource);

            $this->setPrivateKey($data['dh']['priv_key'], self::FORMAT_BINARY);
            $this->setPublicKey($data['dh']['pub_key'], self::FORMAT_BINARY);

            $this->opensslKeyResource = $opensslKeyResource;
        } else {
            // Private key is lazy generated in the absence of ext/openssl
            $publicKey = $this->math->powmod($this->getGenerator(), $this->getPrivateKey(), $this->getPrime());
            $this->setPublicKey($publicKey);
        }

        return $this;
    }

    /**
     * Setter for the value of the public number
     *
     * @throws InvalidArgumentException
     */
    public function setPublicKey(string $number, string $format = self::FORMAT_NUMBER): static
    {
        $number = $this->convert($number, $format, self::FORMAT_NUMBER);
        if (! preg_match('/^\d+$/', $number)) {
            throw new Exception\InvalidArgumentException('Invalid parameter; not a positive natural number');
        }
        $this->publicKey = $number;

        return $this;
    }

    /**
     * Returns own public key for communication to the second party to this transaction
     *
     * @throws InvalidArgumentException
     */
    public function getPublicKey(string $format = self::FORMAT_NUMBER): string
    {
        if ($this->publicKey === null) {
            throw new Exception\InvalidArgumentException(
                'A public key has not yet been generated using a prior call to generateKeys()'
            );
        }

        return $this->convert($this->publicKey, self::FORMAT_NUMBER, $format);
    }

    /**
     * Compute the shared secret key based on the public key received from the
     * the second party to this transaction. This should agree to the secret
     * key the second party computes on our own public key.
     * Once in agreement, the key is known to only to both parties.
     * By default, the function expects the public key to be in binary form
     * which is the typical format when being transmitted.
     *
     * If you need the binary form of the shared secret key, call
     * getSharedSecretKey() with the optional parameter for Binary output.
     *
     * @throws InvalidArgumentException
     * @throws RuntimeException
     */
    public function computeSecretKey(
        string $publicKey,
        string $publicKeyFormat = self::FORMAT_NUMBER,
        string $secretKeyFormat = self::FORMAT_NUMBER
    ): string {
        if (function_exists('openssl_dh_compute_key') && static::$useOpenssl !== false) {
            $publicKey = $this->convert($publicKey, $publicKeyFormat, self::FORMAT_BINARY);
            $secretKey = openssl_dh_compute_key($publicKey, $this->opensslKeyResource);
            if (false === $secretKey) {
                throw new Exception\RuntimeException(
                    'Can not compute key; openssl ' . openssl_error_string()
                );
            }
            $this->secretKey = $this->convert($secretKey, self::FORMAT_BINARY, self::FORMAT_NUMBER);
        } else {
            $publicKey = $this->convert($publicKey, $publicKeyFormat, self::FORMAT_NUMBER);
            if (! preg_match('/^\d+$/', $publicKey)) {
                throw new Exception\InvalidArgumentException(
                    'Invalid parameter; not a positive natural number'
                );
            }
            $this->secretKey = $this->math->powmod($publicKey, $this->getPrivateKey(), $this->getPrime());
        }

        return $this->getSharedSecretKey($secretKeyFormat);
    }

    /**
     * Return the computed shared secret key from the DiffieHellman transaction
     *
     * @throws InvalidArgumentException
     */
    public function getSharedSecretKey(string $format = self::FORMAT_NUMBER): string
    {
        if ($this->secretKey === null) {
            throw new Exception\InvalidArgumentException(
                'A secret key has not yet been computed; call computeSecretKey() first'
            );
        }

        return $this->convert($this->secretKey, self::FORMAT_NUMBER, $format);
    }

    /**
     * Setter for the value of the prime number
     *
     * @throws InvalidArgumentException
     */
    public function setPrime(string $number): static
    {
        if (! preg_match('/^\d+$/', $number) || $number < 11) {
            throw new Exception\InvalidArgumentException(
                'Invalid parameter; not a positive natural number or too small: '
                . 'should be a large natural number prime'
            );
        }
        $this->prime = $number;

        return $this;
    }

    /**
     * Getter for the value of the prime number
     *
     * @throws InvalidArgumentException
     */
    public function getPrime(string $format = self::FORMAT_NUMBER): string
    {
        if ($this->prime === null) {
            throw new Exception\InvalidArgumentException('No prime number has been set');
        }

        return $this->convert($this->prime, self::FORMAT_NUMBER, $format);
    }

    /**
     * Setter for the value of the generator number
     *
     * @throws InvalidArgumentException
     */
    public function setGenerator(string $number): static
    {
        if (! preg_match('/^\d+$/', $number) || $number < 2) {
            throw new Exception\InvalidArgumentException(
                'Invalid parameter; not a positive natural number greater than 1'
            );
        }
        $this->generator = $number;

        return $this;
    }

    /**
     * Getter for the value of the generator number
     *
     * @throws InvalidArgumentException
     */
    public function getGenerator(string $format = self::FORMAT_NUMBER): string
    {
        if ($this->generator === null) {
            throw new Exception\InvalidArgumentException('No generator number has been set');
        }

        return $this->convert($this->generator, self::FORMAT_NUMBER, $format);
    }

    /**
     * Setter for the value of the private number
     *
     * @throws InvalidArgumentException
     */
    public function setPrivateKey(string $number, string $format = self::FORMAT_NUMBER): static
    {
        $number = $this->convert($number, $format, self::FORMAT_NUMBER);
        if (! preg_match('/^\d+$/', $number)) {
            throw new Exception\InvalidArgumentException('Invalid parameter; not a positive natural number');
        }
        $this->privateKey = $number;

        return $this;
    }

    /**
     * Getter for the value of the private number
     */
    public function getPrivateKey(string $format = self::FORMAT_NUMBER): string
    {
        if (! $this->hasPrivateKey()) {
            $this->setPrivateKey($this->generatePrivateKey(), self::FORMAT_BINARY);
        }

        return $this->convert($this->privateKey, self::FORMAT_NUMBER, $format);
    }

    /**
     * Check whether a private key currently exists.
     */
    public function hasPrivateKey(): bool
    {
        return $this->privateKey !== null;
    }

    /**
     * Convert number between formats
     */
    protected function convert(
        string $number,
        string $inputFormat = self::FORMAT_NUMBER,
        string $outputFormat = self::FORMAT_BINARY
    ): string {
        if ($inputFormat === $outputFormat) {
            return $number;
        }

        // convert to number
        switch ($inputFormat) {
            case self::FORMAT_BINARY:
            case self::FORMAT_BTWOC:
                $number = $this->math->binToInt($number);
                break;
            case self::FORMAT_NUMBER:
            default:
                // do nothing
                break;
        }

        // convert to output format
        return match ($outputFormat) {
            self::FORMAT_BINARY => $this->math->intToBin($number),
            self::FORMAT_BTWOC => $this->math->intToBin($number, true),
            default => $number,
        };
    }

    /**
     * In the event a private number/key has not been set by the user,
     * or generated by ext/openssl, a best attempt will be made to
     * generate a random key. Having a random number generator installed
     * on linux/bsd is highly recommended! The alternative is not recommended
     * for production unless without any other option.
     */
    protected function generatePrivateKey(): string
    {
        return Math\Rand::getBytes(mb_strlen($this->getPrime(), '8bit'));
    }
}
