<?php
/**
 * Test WebAPI authentication helper.
 *
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace Magento\Framework\Oauth\Test\Unit\Helper;

use Magento\Framework\App\Request\Http;
use Magento\Framework\Phrase;

class RequestTest extends \PHPUnit_Framework_TestCase
{
    /** @var \Magento\Framework\Oauth\Helper\Request */
    protected $oauthRequestHelper;

    /** @var \Magento\Framework\App\Response\Http */
    protected $response;

    /**
     * @return void
     */
    protected function setUp()
    {
        $this->oauthRequestHelper = new \Magento\Framework\Oauth\Helper\Request();
        $this->response = $this->getMock(
            'Magento\Framework\HTTP\PhpEnvironment\Response',
            ['setHttpResponseCode'],
            [],
            '',
            false
        );
    }

    /**
     * @return void
     */
    protected function tearDown()
    {
        unset($this->oauthRequestHelper, $this->response);
    }

    /**
     * @param \Exception $exception
     * @param array $expected
     * @return void
     * @dataProvider dataProviderForPrepareErrorResponseTest
     */
    public function testPrepareErrorResponse($exception, $expected)
    {
        $this->response
            ->expects($this->once())
            ->method('setHttpResponseCode')
            ->with($expected[1]);

        $errorResponse = $this->oauthRequestHelper->prepareErrorResponse($exception, $this->response);
        $this->assertEquals(['oauth_problem' => $expected[0]], $errorResponse);
    }

    /**
     * @return array
     */
    public function dataProviderForPrepareErrorResponseTest()
    {
        return [
            [
                new \Magento\Framework\Oauth\OauthInputException(new Phrase('msg')),
                ['msg', \Magento\Framework\Oauth\Helper\Request::HTTP_BAD_REQUEST],
            ],
            [
                new \Exception('msg'),
                ['internal_error&message=msg', \Magento\Framework\Oauth\Helper\Request::HTTP_INTERNAL_ERROR]
            ],
            [
                new \Exception(),
                [
                    'internal_error&message=empty_message',
                    \Magento\Framework\Oauth\Helper\Request::HTTP_INTERNAL_ERROR
                ]
            ]
        ];
    }

    /**
     * @param string $url
     * @param string $host
     * @return void
     * @dataProvider hostsDataProvider
     */
    public function testGetRequestUrl($url, $host)
    {
        $httpRequestMock = $this->getMock(
            'Magento\Framework\App\Request\Http',
            ['getHttpHost', 'getScheme', 'getRequestUri'],
            [],
            '',
            false
        );

        $httpRequestMock->expects($this->any())->method('getHttpHost')->will($this->returnValue($host));
        $httpRequestMock->expects($this->any())->method('getScheme')->will($this->returnValue('http'));
        $httpRequestMock->expects($this->any())->method('getRequestUri')->will($this->returnValue('/'));

        $this->assertEquals($url, $this->oauthRequestHelper->getRequestUrl($httpRequestMock));
    }

    /**
     * @return array
     */
    public function hostsDataProvider()
    {
        return  [
            'hostWithoutPort' => [
                'url' => 'http://localhost/',
                'host' => 'localhost'
            ],
            'hostWithPort' => [
                'url' => 'http://localhost:81/',
                'host' => 'localhost:81'
            ]
        ];
    }

    /**
     * Test that the OAuth parameters are correctly extracted from the Authorization header.
     *
     * @param $authHeaderValue
     * @param $expectedParams
     * @dataProvider dataProviderForTestPrepareRequestOAuthHeader
     */
    public function testPrepareRequestOAuthHeader($authHeaderValue, $expectedParams)
    {
        $httpRequestMock = $this->getMockBuilder(Http::class)
            ->disableOriginalConstructor()
            ->getMock();

        $httpRequestMock->expects($this->once())->method('getScheme')->willReturn('https');
        $httpRequestMock->expects($this->once())->method('getHttpHost')->willReturn('example.com');
        $httpRequestMock->expects($this->once())->method('getRequestUri')->willReturn('/');

        $httpRequestMock->expects($this->any())
            ->method('getHeader')
            ->willReturnCallback(function ($header) use ($authHeaderValue) {
                switch ($header) {
                    case 'Authorization':
                        return $authHeaderValue;
                    case \Zend_Http_Client::CONTENT_TYPE:
                        return \Zend_Http_Client::ENC_URLENCODED;
                    default:
                        return null;
                }
            });

        $this->assertEquals($expectedParams, $this->oauthRequestHelper->prepareRequest($httpRequestMock));
    }

    /**
     * @return array
     */
    public function dataProviderForTestPrepareRequestOAuthHeader()
    {
        return [
            [
                null,
                []
            ],
            [
                '',
                []
            ],
            [
                'OAuth oauth_consumer_key="x",oauth_token="x", Basic d2luZHNvcm0yOldpTmRzb1JTbWlUSDAwMTQ=',
                ['oauth_consumer_key' => 'x', 'oauth_token' => 'x']
            ],
            [
                'Basic d2luZHNvcm0yOldpTmRzb1JTbWlUSDAwMTQ=, OAuth oauth_consumer_key="x",oauth_token="x"',
                ['oauth_consumer_key' => 'x', 'oauth_token' => 'x']
            ],
            [
                'Basic d2luZHNvcm0yOldpTmRzb1JTbWlUSDAwMTQ=, oauth oauth_consumer_key="x", oauth_token="x"',
                ['oauth_consumer_key' => 'x', 'oauth_token' => 'x']
            ],
            [
                'oauth oauth_consumer_key="x", oauth_token="x", Basic d2luZHNvcm0yOldpTmRzb1JTbWlUSDAwMTQ=',
                ['oauth_consumer_key' => 'x', 'oauth_token' => 'x']
            ]
        ];
    }
}
