<?php
/**
 * This file is part of the Klarna Kp module
 *
 * (c) Klarna AB
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */

namespace Klarna\Kp\Plugin\Payment\Helper;

use Klarna\Core\Exception as KlarnaException;
use Klarna\Kp\Api\QuoteRepositoryInterface;
use Klarna\Kp\Model\Payment\Kp;
use Klarna\Kp\Model\Session;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Payment\Model\Method\Factory;
use Magento\Payment\Model\MethodInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Psr\Log\LoggerInterface;

class DataPlugin
{
    /**
     * @var CartRepositoryInterface
     */
    private $mageQuoteRepository;

    /**
     * @var LoggerInterface
     */
    private $log;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var Session
     */
    private $session;

    /**
     * Factory for payment method models
     *
     * @var Factory
     */
    private $methodFactory;

    /**
     * @var \Magento\Framework\Registry
     */
    private $coreRegistry;

    /**
     * DataPlugin constructor.
     *
     * @param \Magento\Framework\Registry $coreRegistry
     * @param CartRepositoryInterface     $mageQuoteRepository
     * @param Session                     $session
     * @param QuoteRepositoryInterface    $quoteRepository
     * @param Factory                     $paymentMethodFactory
     * @param LoggerInterface             $log
     */
    public function __construct(
        \Magento\Framework\Registry $coreRegistry,
        CartRepositoryInterface $mageQuoteRepository,
        Session $session,
        QuoteRepositoryInterface $quoteRepository,
        Factory $paymentMethodFactory,
        LoggerInterface $log
    ) {
        $this->session = $session;
        $this->quoteRepository = $quoteRepository;
        $this->methodFactory = $paymentMethodFactory;
        $this->log = $log;
        $this->coreRegistry = $coreRegistry;
        $this->mageQuoteRepository = $mageQuoteRepository;
    }

    /**
     * Modify results of getPaymentMethods() call to add in Klarna methods returned by API
     *
     * @param \Magento\Payment\Helper\Data $subject
     * @param callable                     $proceed
     * @return array
     */
    public function aroundGetPaymentMethods(\Magento\Payment\Helper\Data $subject, callable $proceed) {
        $result = $proceed();
        $quote = null;
        $methods = $this->getKlarnaMethods();
        foreach ($methods as $method) {
            $result[$method] = $result['klarna_kp'];
            unset($result[$method]['title']);
        }
        if (count($methods)) {
            unset($result['klarna_kp']);
        }
        return $result;
    }

    /**
     * @return string[]
     */
    private function getKlarnaMethods()
    {
        $quote = null;
        try {
            /** @var \Magento\Sales\Api\Data\OrderInterface $order */
            if ($order = $this->coreRegistry->registry('current_order')) {
                $quote = $this->mageQuoteRepository->get($order->getQuoteId());
            }
            if (!$quote) {
                $quote = $this->session->getQuote();
            }
            return $this->processQuote($quote);
        } catch (NoSuchEntityException $e) {
            return Kp::KLARNA_METHODS;
        }
    }

    /**
     * @param CartInterface $quote
     * @return string[]
     */
    private function processQuote(CartInterface $quote)
    {
        try {
            return $this->quoteRepository->getActiveByQuote($quote)->getPaymentMethods();
        } catch (KlarnaException $e) {
            $this->log->error($e);
        }
        return Kp::KLARNA_METHODS;
    }

    /**
     * Modify results of getMethodInstance() call to add in details about Klarna payment method
     *
     * @param \Magento\Payment\Helper\Data $subject
     * @param callable                     $proceed
     * @param string                       $code
     * @return MethodInterface
     */
    public function aroundGetMethodInstance(\Magento\Payment\Helper\Data $subject, callable $proceed, $code)
    {
        try {
            $methods = $this->getKlarnaMethods();
            if (!in_array($code, $methods)) {
                return $proceed($code);
            }
            /** @var \Klarna\Kp\Model\Payment\Kp $methodInstance */
            $methodInstance = $this->methodFactory->create('\Klarna\Kp\Model\Payment\Kp');
            $methodInstance->setCode($code);
            return $methodInstance;
        } catch (NoSuchEntityException $e) {
            return $proceed($code);
        }
    }

}
