<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */
namespace Klarna\Kp\Model\Api\Builder;

use Klarna\Core\Exception as KlarnaException;
use Klarna\Core\Helper\ConfigHelper;
use Klarna\Core\Model\Api\Exception as KlarnaApiException;
use Klarna\Core\Model\Checkout\Orderline\Collector;
use Klarna\Kp\Api\Data\RequestInterface;
use Klarna\Kp\Model\Api\Request;
use Klarna\Kp\Model\Payment\Kp;
use Magento\Directory\Helper\Data as DirectoryHelper;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DataObject;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\ObjectManagerInterface as ObjectManager;
use Magento\Framework\Url;
use Magento\Quote\Model\Quote as MageQuote;
use Magento\Quote\Model\Quote\Address;
use Magento\Quote\Model\Quote\Address as QuoteAddress;
use Magento\Store\Api\Data\StoreInterface;

class Kasper extends \Klarna\Core\Model\Api\Builder
{
    public function __construct(
        EventManager $eventManager,
        Collector $collector,
        Url $url,
        ConfigHelper $configHelper,
        ScopeConfigInterface $config,
        DirectoryHelper $directoryHelper,
        \Magento\Framework\Stdlib\DateTime\DateTime $coreDate,
        \Magento\Framework\Stdlib\DateTime $dateTime,
        ObjectManager $objManager,
        array $data = []
    ) {
        parent::__construct(
            $eventManager,
            $collector,
            $url,
            $configHelper,
            $config,
            $directoryHelper,
            $coreDate,
            $dateTime,
            $objManager,
            $data
        );
        $this->prefix = 'kp';
    }

    /**
     * Generate request
     *
     * @param string $type
     * @return $this
     */
    public function generateRequest($type = self::GENERATE_TYPE_CREATE)
    {
        $this->resetOrderLines();
        parent::generateRequest($type);

        switch ($type) {
            case self::GENERATE_TYPE_CREATE:
            case self::GENERATE_TYPE_UPDATE:
                return $this->_generateCreateUpdate();
            case self::GENERATE_TYPE_PLACE:
                return $this->_generatePlace();
        }
        return $this;
    }

    /**
     * Generate Create/Update Request
     *
     * @return $this
     */
    protected function _generateCreateUpdate()
    {
        $requiredAttributes = [
            'purchase_country',
            'purchase_currency',
            'locale',
            'order_amount',
            'order_lines'
        ];

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->getObject();
        $store = $quote->getStore();
        $create = [
            'purchase_country'  => $this->directoryHelper->getDefaultCountry($store),
            'purchase_currency' => $quote->getBaseCurrencyCode(),
            'locale'            => str_replace('_', '-', $this->configHelper->getLocaleCode()),
        ];

        /**
         * Pre-fill customer details
         */
        if ($this->configHelper->getPaymentConfigFlag('data_sharing', $store, Kp::METHOD_CODE) && ($this->configHelper->getApiConfig('api_version', $store) == 'kp_na')) {
            $create = $this->prefill($create, $quote, $store);
        }

        /**
         * External payment methods
         */
        $create['external_payment_methods'] = $this->getExternalMethods(
            $this->configHelper->getPaymentConfig('external_payment_methods', $store, Kp::METHOD_CODE),
            $store
        );

        /**
         * Options
         */
        $create['options'] = $this->getOptions($store);

        /**
         * Totals
         */
        $address = $quote->isVirtual() ? $quote->getBillingAddress() : $quote->getShippingAddress();
        $create['order_amount'] = $this->configHelper->toApiFloat($address->getBaseGrandTotal());

        $create['order_lines'] = $this->getOrderLines();
        $create['order_tax_amount'] = $this->configHelper->toApiFloat($address->getBaseTaxAmount());

        if ($shippingCountries = $this->configHelper->getPaymentConfig('shipping_countries', $store)) {
            $create['shipping_countries'] = explode(',', $shippingCountries);
        }

        /**
         * Merchant reference
         */
        $merchantReferences = $this->getMerchantReferences($quote);

        if ($merchantReferences->getData('merchant_reference_1')) {
            $create['merchant_reference1'] = $merchantReferences->getData('merchant_reference_1');
        }

        if (!empty($merchantReferences['merchant_reference_2'])) {
            $create['merchant_reference2'] = $merchantReferences->getData('merchant_reference_2');
        }

        /**
         * Urls
         */
        $urlParams = [
            '_nosid'         => true,
            '_forced_secure' => true
        ];

        $create['merchant_urls'] = $this->processMerchantUrls($store, $urlParams);

        $create = $this->validateRequest($create, self::GENERATE_TYPE_CREATE, $requiredAttributes);

        $this->setRequest($create, self::GENERATE_TYPE_CREATE);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function prefill($create, $quote, $store)
    {
        $create = parent::prefill($create, $quote, $store);
        foreach (['billing_address', 'shipping_address'] as $type) {
            if (!isset($create[$type])) {
                continue; // If address not set, skip
            }
            if (!isset($create[$type]['email'])) {
                unset($create[$type]); // Address isn't 'valid' unless it has an email
            }
        }
        return $create;
    }

    /**
     * Pre-process Merchant URLs
     *
     * @param StoreInterface $store
     * @param string[] $urlParams
     * @return string[]
     */
    public function processMerchantUrls($store, $urlParams)
    {
        $merchantUrls = new DataObject([
            'confirmation' => $this->url->getDirectUrl('checkout/onepage/success', $urlParams),
            'notification' => $this->url->getDirectUrl('klarna/api/disabled', $urlParams)
        ]);

        $this->eventManager->dispatch(
            'klarna_prepare_merchant_urls',
            [
                'urls'       => $merchantUrls,
                'url_params' => new DataObject($urlParams)
            ]
        );
        $data = $merchantUrls->toArray();

        $data['notification'] = preg_replace('/\/id\/{checkout\.order\.id}/', '', $data['notification']);
        unset($data['push']);

        return $data;
    }

    /**
     * Validate that request has all required fields
     *
     * @param string[][] $create
     * @param string     $type
     * @param string[]   $requiredAttributes
     * @return \string[][]
     */
    protected function validateRequest($create, $type, $requiredAttributes)
    {
        $create = array_filter($create);

        $missingAttributes = [];
        foreach ($requiredAttributes as $requiredAttribute) {
            if (empty($create[$requiredAttribute])) {
                $missingAttributes[] = $requiredAttribute;
            }
        }
        if (!empty($missingAttributes)) {
            throw new KlarnaApiException(__(
                'Missing required attribute(s) on %1: "%2".',
                $type,
                implode(', ', $missingAttributes)
            ));
        }
        return $create;
    }

    /**
     * Generate place order body
     *
     * @return Kasper
     * @throws \Klarna\Core\Exception
     * @throws \Klarna\Core\Model\Api\Exception
     */
    protected function _generatePlace()
    {
        $requiredAttributes = [
            'purchase_country',
            'purchase_currency',
            'locale',
            'order_amount',
            'order_lines',
            'merchant_urls',
            'billing_address',
            'shipping_address'
        ];

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->getObject();
        $store = $quote->getStore();
        $create = [
            'purchase_country'  => $this->directoryHelper->getDefaultCountry($store),
            'purchase_currency' => $quote->getBaseCurrencyCode(),
            'locale'            => str_replace('_', '-', $this->configHelper->getLocaleCode()),
        ];

        /**
         * Get customer addresses (shipping and billing)
         */
        $create['billing_address'] = $this->_getAddressData($quote, Address::TYPE_BILLING);
        $create['shipping_address'] = $this->_getAddressData($quote, Address::TYPE_SHIPPING);

        /**
         * Totals
         */
        $address = $quote->isVirtual() ? $quote->getBillingAddress() : $quote->getShippingAddress();
        $create['order_amount'] = (int)$this->configHelper->toApiFloat($address->getBaseGrandTotal());
        $create['order_lines'] = $this->getOrderLines();
        $create['order_tax_amount'] = $this->configHelper->toApiFloat($address->getBaseTaxAmount());

        /**
         * Merchant reference
         */
        $merchantReferences = $this->getMerchantReferences($quote);

        if ($merchantReferences->getData('merchant_reference_1')) {
            $create['merchant_reference1'] = $merchantReferences->getData('merchant_reference_1');
        }

        if (!empty($merchantReferences['merchant_reference_2'])) {
            $create['merchant_reference2'] = $merchantReferences->getData('merchant_reference_2');
        }

        /**
         * Urls
         */
        $urlParams = [
            '_nosid'         => true,
            '_forced_secure' => true
        ];

        $create['merchant_urls'] = $this->processMerchantUrls($store, $urlParams);

        $create = $this->validateRequest($create, self::GENERATE_TYPE_PLACE, $requiredAttributes);

        $total = 0;
        foreach ($create['order_lines'] as $orderLine) {
            $total += (int)$orderLine['total_amount'];
        }

        if ($total !== $create['order_amount']) {
            throw new KlarnaException(__('Order line totals do not total order_amount'));
        }

        $this->setRequest($create, self::GENERATE_TYPE_PLACE);

        return $this;
    }

    /**
     * Get request
     *
     * @return Request
     */
    public function getRequest()
    {
        /** @var string[][] $request */
        $request = parent::getRequest();
        if ($request instanceof RequestInterface) {
            $request = $request->toArray();
        }
        if (isset($request['customer'])) {
            $request['customer'] = new Request\Customer($request['customer']);
        }
        if (isset($request['options'])) {
            $request['options'] = new Request\Options($request['options']);
        }
        if (isset($request['order_lines'])) {
            foreach ($request['order_lines'] as $key => $orderLine) {
                $request['order_lines'][$key] = new Request\Orderline($orderLine);
            }
        }
        if (isset($request['urls'])) {
            $request['urls'] = new Request\MerchantUrls($request['urls']);
        }
        if (isset($request['merchant_urls'])) {
            $request['urls'] = new Request\MerchantUrls($request['merchant_urls']);
            unset($request['merchant_urls']);
        }
        if (isset($request['attachment'])) {
            $request['attachment'] = new Request\Attachment($request['attachment']);
        }
        if (isset($request['billing_address'])) {
            $request['billing_address'] = new Request\Address($request['billing_address']);
        }
        if (isset($request['shipping_address'])) {
            $request['shipping_address'] = new Request\Address($request['shipping_address']);
        }
        return new Request($request);
    }

    /**
     * Get available shipping methods for a quote for the api init
     *
     * @param MageQuote $quote
     *
     * @return array
     */
    protected function _getShippingMethods(MageQuote $quote)
    {
        $rates = [];
        if ($quote->isVirtual()) {
            return $rates;
        }

        /** @var \Magento\Quote\Model\Quote\Address\Rate $rate */
        foreach ($quote->getShippingAddress()->getAllShippingRates() as $rate) {
            if (!$rate->getCode() || !$rate->getMethodTitle()) {
                continue;
            }

            $rates[] = [
                'id'          => $rate->getCode(),
                'name'        => $rate->getMethodTitle(),
                'price'       => $this->configHelper->toApiFloat($rate->getPrice()),
                'promo'       => '',
                'tax_amount'  => 0,
                'tax_rate'    => 0,
                'description' => $rate->getMethodDescription(),
                'preselected' => $rate->getCode() == $quote->getShippingAddress()->getShippingMethod()
            ];
        }
        return $this->configHelper->removeDuplicates($rates);
    }
}
