<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */
namespace Klarna\Kp\Model;

use Klarna\Kp\Api\CreditApiInterface;
use Klarna\Kp\Api\QuoteInterface;
use Klarna\Kp\Api\QuoteRepositoryInterface;
use Klarna\Kp\Model\QuoteRepository\SaveHandler;
use Klarna\Kp\Model\ResourceModel\Quote as QuoteResource;
use Klarna\Kp\Model\ResourceModel\Quote\CollectionFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\Data\CartInterface as MageQuoteInterface;

/**
 * Class QuoteRepository
 *
 * @package Klarna\Kp\Model
 */
class QuoteRepository implements QuoteRepositoryInterface
{
    /**
     * @var QuoteFactory
     */
    protected $quoteFactory;

    /**
     * @var QuoteResource
     */
    protected $resourceModel;

    /**
     * Holds a cache of instances to avoid unnecessary db and API calls
     *
     * @var array
     */
    protected $instancesById = [];

    /**
     * Holds a cache of instances to avoid unnecessary db and API calls
     *
     * @var array
     */
    protected $instances = [];

    /**
     * @var SaveHandler
     */
    protected $saveHandler;

    /**
     * @var CreditApiInterface
     */
    protected $api;

    /**
     * QuoteRepository constructor.
     *
     * @param QuoteFactory       $quoteFactory
     * @param QuoteResource      $resourceModel
     * @param CollectionFactory  $collectionFactory
     * @param SaveHandler        $saveHandler
     * @param CreditApiInterface $api
     * @codeCoverageIgnore
     */
    public function __construct(
        QuoteFactory $quoteFactory,
        QuoteResource $resourceModel,
        SaveHandler $saveHandler,
        CreditApiInterface $api
    ) {
        $this->quoteFactory = $quoteFactory;
        $this->resourceModel = $resourceModel;
        $this->saveHandler = $saveHandler;
        $this->api = $api;
    }

    /**
     * Load quote with different methods
     *
     * @param string $loadMethod
     * @param string $loadField
     * @param int    $identifier
     * @throws NoSuchEntityException
     * @return QuoteInterface
     */
    public function loadQuote($loadMethod, $loadField, $identifier)
    {
        /** @var QuoteInterface $quote */
        $quote = $this->quoteFactory->create();
        $quote->$loadMethod($identifier, $loadField);
        if (!$quote->getId()) {
            throw NoSuchEntityException::singleField($loadField, $identifier);
        }
        return $quote;
    }

    /**
     * Cache instance locally in memory to avoid additional DB calls
     *
     * @param QuoteInterface $quote
     */
    protected function cacheInstance(QuoteInterface $quote)
    {
        $this->instancesById[$quote->getId()] = $quote;
        $this->instances[$quote->getSessionId()] = $quote;
    }

    /**
     * Get quote by Magento quote
     *
     * @param MageQuoteInterface $mageQuote
     * @return QuoteInterface
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getActiveByQuote(MageQuoteInterface $mageQuote)
    {
        $quoteId = $this->resourceModel->getActiveByQuote($mageQuote);
        if (!$quoteId) {
            throw NoSuchEntityException::singleField('quote_id', $mageQuote->getId());
        }
        return $this->loadQuote('load', 'payments_quote_id', $quoteId);
    }

    /**
     * Delete quote by ID
     *
     * @param int $id
     * @return void
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function deleteById($id)
    {
        $this->delete($this->getById($id));
    }

    /**
     * Delete quote
     *
     * @param QuoteInterface $quote
     * @return void
     */
    public function delete(QuoteInterface $quote)
    {
        $quoteId = $quote->getId();
        $clientToken = $quote->getClientToken();
        $sessionId = $quote->getSessionId();
        $this->api->cancelOrder($sessionId);
        $quote->delete();
        unset($this->instances[$clientToken]);
        unset($this->instances[$sessionId]);
        unset($this->instancesById[$quoteId]);
    }

    /**
     * Get quote by ID
     *
     * @param int  $quoteId
     * @param bool $forceReload
     * @return QuoteInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getById($quoteId, $forceReload = false)
    {
        if (!isset($this->instancesById[$quoteId]) || $forceReload) {
            /** @var QuoteInterface $quote */
            $quote = $this->loadQuote('load', 'payments_quote_id', $quoteId);
            $this->cacheInstance($quote);
        }
        return $this->instancesById[$quoteId];
    }

    /**
     * Mark quote as inactive and cancel it with API
     *
     * @param QuoteInterface $quote
     */
    public function markInactive(QuoteInterface $quote)
    {
        $quote->setIsActive(0);
        $this->save($quote);

        if ($quote->getAuthorizationToken()) {
            $this->api->cancelOrder($quote->getAuthorizationToken());
        }
    }

    /**
     * Save Klarna Quote
     *
     * @param QuoteInterface $quote
     * @return \Klarna\Kp\Api\QuoteInterface
     */
    public function save(QuoteInterface $quote)
    {
        return $this->saveHandler->save($quote);
    }

    /**
     * Load quote by session_id
     *
     * @param string $sessionId
     * @param bool   $forceReload
     * @return QuoteInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getBySessionId($sessionId, $forceReload = false)
    {
        if ($forceReload || !isset($this->instances[$sessionId])) {
            $quote = $this->loadQuote('load', 'session_id', $sessionId);
            $this->cacheInstance($quote);
        }
        return $this->instances[$sessionId];
    }
}
