<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */
namespace Klarna\Kp\Model\Payment;

use Klarna\Core\Helper\ConfigHelper;
use Klarna\Core\Model\Api\Exception as KlarnaApiException;
use Klarna\Core\Exception as KlarnaException;
use Klarna\Kp\Api\Data\ResponseInterface;
use Klarna\Kp\Api\QuoteRepositoryInterface;
use Klarna\Kp\Model\QuoteFactory;
use Klarna\Kp\Model\Session;
use Magento\Framework\DataObject;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Payment\Model\InfoInterface;
use Magento\Payment\Model\Method\Adapter;
use Magento\Payment\Model\MethodInterface;
use Magento\Quote\Api\Data\CartInterface;
use Psr\Log\LoggerInterface;

/**
 * Klarna Payment
 */
class Kp implements MethodInterface
{
    const METHOD_CODE = 'klarna_kp';

    const ACTION_ORDER             = 'order';
    const ACTION_AUTHORIZE         = 'authorize';
    const ACTION_AUTHORIZE_CAPTURE = 'authorize_capture';

    /**
     * @var Session
     */
    protected $session;

    /**
     * @var Adapter
     */
    protected $adapter;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * @var EventManager
     */
    protected $eventManager;

    /**
     * @var LoggerInterface
     */
    protected $log;

    /**
     * @var QuoteFactory
     */
    protected $quoteFactory;

    /**
     * @var QuoteRepositoryInterface
     */
    private $quoteRepository;

    /**
     * Kco constructor.
     *
     * @param Adapter                  $adapter
     * @param Session                  $session
     * @param QuoteRepositoryInterface $quoteRepository
     * @param QuoteFactory             $quoteFactory
     * @param ConfigHelper             $configHelper
     * @param EventManager             $eventManager
     * @param LoggerInterface          $logger
     * @internal param \Magento\Checkout\Model\Session $checkoutSession
     */
    public function __construct(
        Adapter $adapter,
        Session $session,
        QuoteRepositoryInterface $quoteRepository,
        QuoteFactory $quoteFactory,
        ConfigHelper $configHelper,
        EventManager $eventManager,
        LoggerInterface $logger
    ) {
        $this->adapter = $adapter;
        $this->session = $session;
        $this->configHelper = $configHelper;
        $this->eventManager = $eventManager;
        $this->log = $logger;
        $this->quoteRepository = $quoteRepository;
        $this->quoteFactory = $quoteFactory;
    }

    /**
     * {@inheritdoc}
     */
    public function isActive($storeId = null)
    {
        return $this->adapter->isActive($storeId);
    }

    /**
     * {@inheritdoc}
     */
    public function getCode()
    {
        return $this->adapter->getCode();
    }

    /**
     * {@inheritdoc}
     */
    public function getFormBlockType()
    {
        return $this->adapter->getFormBlockType();
    }

    /**
     * {@inheritdoc}
     */
    public function getTitle()
    {
        return $this->adapter->getTitle();
    }

    /**
     * {@inheritdoc}
     */
    public function setStore($storeId)
    {
        $this->adapter->setStore($storeId);
    }

    /**
     * {@inheritdoc}
     */
    public function getStore()
    {
        return $this->adapter->getStore();
    }

    /**
     * {@inheritdoc}
     */
    public function canOrder()
    {
        return $this->adapter->canOrder();
    }

    /**
     * {@inheritdoc}
     */
    public function canAuthorize()
    {
        return $this->adapter->canAuthorize();
    }

    /**
     * {@inheritdoc}
     */
    public function canCapture()
    {
        return $this->adapter->canCapture();
    }

    /**
     * {@inheritdoc}
     */
    public function canCapturePartial()
    {
        return $this->adapter->canCapturePartial();
    }

    /**
     * {@inheritdoc}
     */
    public function canCaptureOnce()
    {
        return $this->adapter->canCaptureOnce();
    }

    /**
     * {@inheritdoc}
     */
    public function canRefund()
    {
        return $this->adapter->canRefund();
    }

    /**
     * {@inheritdoc}
     */
    public function canRefundPartialPerInvoice()
    {
        return $this->adapter->canRefundPartialPerInvoice();
    }

    /**
     * {@inheritdoc}
     */
    public function canVoid()
    {
        return $this->adapter->canVoid();
    }

    /**
     * {@inheritdoc}
     */
    public function canUseInternal()
    {
        return $this->adapter->canUseInternal();
    }

    /**
     * {@inheritdoc}
     */
    public function canUseCheckout()
    {
        return $this->adapter->canUseCheckout();
    }

    /**
     * {@inheritdoc}
     */
    public function canEdit()
    {
        return $this->adapter->canEdit();
    }

    /**
     * {@inheritdoc}
     */
    public function canFetchTransactionInfo()
    {
        return $this->adapter->canFetchTransactionInfo();
    }

    /**
     * {@inheritdoc}
     */
    public function fetchTransactionInfo(InfoInterface $payment, $transactionId)
    {
        return $this->adapter->fetchTransactionInfo($payment, $transactionId);
    }

    /**
     * {@inheritdoc}
     */
    public function isGateway()
    {
        return $this->adapter->isGateway();
    }

    /**
     * {@inheritdoc}
     */
    public function isOffline()
    {
        return $this->adapter->isOffline();
    }

    /**
     * {@inheritdoc}
     */
    public function isInitializeNeeded()
    {
        return $this->adapter->isInitializeNeeded();
    }

    /**
     * {@inheritdoc}
     */
    public function canUseForCountry($country)
    {
        return $this->adapter->canUseForCountry($country);
    }

    /**
     * {@inheritdoc}
     */
    public function canUseForCurrency($currencyCode)
    {
        return $this->adapter->canUseForCurrency($currencyCode);
    }

    /**
     * {@inheritdoc}
     */
    public function getInfoBlockType()
    {
        return $this->adapter->getInfoBlockType();
    }

    /**
     * {@inheritdoc}
     */
    public function getInfoInstance()
    {
        return $this->adapter->getInfoInstance();
    }

    /**
     * {@inheritdoc}
     */
    public function setInfoInstance(InfoInterface $info)
    {
        $this->adapter->setInfoInstance($info);
    }

    /**
     * {@inheritdoc}
     */
    public function validate()
    {
        $this->adapter->validate();
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function order(InfoInterface $payment, $amount)
    {
        $this->adapter->order($payment, $amount);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function authorize(InfoInterface $payment, $amount)
    {
        $this->adapter->authorize($payment, $amount);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function capture(InfoInterface $payment, $amount)
    {
        $this->adapter->capture($payment, $amount);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function refund(InfoInterface $payment, $amount)
    {
        $this->adapter->refund($payment, $amount);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function cancel(InfoInterface $payment)
    {
        $this->adapter->cancel($payment);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function void(InfoInterface $payment)
    {
        $this->adapter->void($payment);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function canReviewPayment()
    {
        return $this->adapter->canReviewPayment();
    }

    /**
     * {@inheritdoc}
     */
    public function acceptPayment(InfoInterface $payment)
    {
        $this->adapter->acceptPayment($payment);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function denyPayment(InfoInterface $payment)
    {
        $this->adapter->denyPayment($payment);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getConfigData($field, $storeId = null)
    {
        return $this->adapter->getConfigData($field, $storeId);
    }

    /**
     * {@inheritdoc}
     */
    public function assignData(DataObject $data)
    {
        $this->adapter->assignData($data);
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function isAvailable(CartInterface $quote = null)
    {
        $available = $this->adapter->isAvailable($quote);
        if (!$available) {
            return false;
        }

        if (!$quote) {
            $quote = $this->session->getQuote();
        }

        if (!$quote) {
            return false;
        }

        if (!in_array($this->configHelper->getApiConfig('api_version', $quote->getStore()), ['kp_na', 'kp_eu'], true)) {
            return false;
        }

        try {
            $klarnaCheckoutId = $this->quoteRepository->getActiveByQuote($quote)->getSessionId();
        } catch (NoSuchEntityException $e) {
            $klarnaCheckoutId = '';
        }
        /** @var ResponseInterface $klarnaPayments */
        try {
            $klarnaPayments = $this->session->init($klarnaCheckoutId);
            return $klarnaPayments->getIsSuccessfull();
        } catch (KlarnaApiException $e) {
            $this->log->error($e);
            return false;
        } catch (KlarnaException $e) {
            $this->log->error($e);
            return true;
        }
    }

    /**
     * {@inheritdoc}
     */
    public function initialize($paymentAction, $stateObject)
    {
        return $this->adapter->initialize($paymentAction, $stateObject);
    }

    /**
     * {@inheritdoc}
     */
    public function getConfigPaymentAction()
    {
        return $this->adapter->getConfigPaymentAction();
    }
}
