<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */

namespace Klarna\Kp\Model\Api\Rest\Service;

use GuzzleHttp\Client;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\MessageFormatter;
use GuzzleHttp\Middleware;
use Klarna\Core\Helper\ConfigHelper;
use Klarna\Core\Helper\VersionInfo;
use Klarna\Core\Model\Api\Exception as KlarnaApiException;
use Klarna\Kp\Api\CreditApiInterface;
use Klarna\Kp\Api\Data\RequestInterface;
use Klarna\Kp\Api\Data\ResponseInterface;
use Klarna\Kp\Model\Api\Response;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Module\ResourceInterface;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Payments
 *
 * @package Klarna\Kp\Model\Api\Rest\Service
 */
class Payments implements CreditApiInterface
{
    const API_VERSION = 'v1';
    const LOG_FORMAT  = ">>>>>>>> REQUEST:\n{request}\n<<<<<<<< RESPONSE:\n{response}\n-------- ERRORS:\n{error}\n++++++++\n";

    /**
     * @var LoggerInterface $log
     */
    protected $log;

    /**
     * @var string
     */
    protected $username;

    /**
     * @var string
     */
    protected $password;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ScopeConfigInterface
     */
    protected $config;

    /**
     * @var StoreInterface
     */
    protected $store;

    /**
     * @var ConfigHelper
     */
    protected $configHelper;

    /**
     * Holds headers to be sent in HTTP request
     *
     * @var array
     */
    protected $headers = [];

    /**
     * Kasper constructor.
     *
     * @param ConfigHelper      $configHelper
     * @param StoreInterface    $store
     * @param ResourceInterface $moduleResource
     * @param LoggerInterface   $log
     * @param VersionInfo       $versionInfo
     */
    public function __construct(
        ConfigHelper $configHelper,
        StoreManagerInterface $storeManager,
        ResourceInterface $moduleResource,
        LoggerInterface $log,
        VersionInfo $versionInfo
    ) {
        $this->log = $log;
        $stack = HandlerStack::create();
        $stack->push(
            Middleware::log(
                $this->log,
                new MessageFormatter(self::LOG_FORMAT)
            )
        );
        $this->client = new Client(
            [
                'handler' => $stack,
            ]
        );
        $this->configHelper = $configHelper;
        $this->store = $storeManager->getStore();
        $this->username = $this->configHelper->getApiConfig('merchant_id', $this->store);
        $this->password = $this->configHelper->getApiConfig('shared_secret', $this->store);
        $version = $versionInfo->getVersion('klarna/module-kp');
        $version .= ';Core/' . $versionInfo->getVersion('klarna/module-core');
        $version .= ';OM/' . $versionInfo->getVersion('klarna/module-om');
        $mageMode = $versionInfo->getMageMode();
        $mageVersion = $versionInfo->getMageEdition() . '/' . $versionInfo->getMageVersion();
        $baseUA = 'Guzzle/' . \GuzzleHttp\Client::VERSION . ';PHP';
        $this->setUserAgent($baseUA, PHP_VERSION, null, null, false);
        $this->setUserAgent('Magento2_KP', $version, $mageVersion, $mageMode, true);
        $this->setHeader('Accept', '*/*');
    }

    /**
     * {@inheritdoc}
     */
    public function setUserAgent($product, $version, $mageVersion, $mageMode, $keepCurrent = true)
    {
        if (!isset($this->headers['User-Agent'])) {
            $this->headers['User-Agent'] = '';
        }
        $current = ';' . $this->headers['User-Agent'];
        if (!$keepCurrent) {
            $current = '';
        }
        $magento = '';
        if ($mageVersion !== null) {
            $magento = '(Magento ' . $mageVersion  . ';' . $mageMode . ' mode)';
        }
        $this->setHeader(
            'User-Agent',
            $product . '/' . $version . ';' . $magento . $current
        );
    }

    /**
     * {@inheritdoc}
     */
    public function setHeader($header, $value = null)
    {
        if (!$value) {
            unset($this->headers[$header]);
            return;
        }
        $this->headers[$header] = $value;
    }

    /**
     * @param RequestInterface $request
     * @return ResponseInterface
     */
    public function createSession(RequestInterface $request)
    {
        return $this->processRequest('/credit/' . self::API_VERSION . '/sessions', $request);
    }

    /**
     * @param string           $url
     * @param RequestInterface $request
     * @param string           $method
     * @return Response
     */
    protected function processRequest($url, RequestInterface $request = null, $method = 'post')
    {
        $this->setHeader('Content-Type', 'application/json');
        $data = [
            'headers' => $this->headers,
            'auth'    => [$this->username, $this->password],
        ];
        if ($request) {
            $data['body'] = json_encode($request->toArray());
        }
        $response = [];
        try {
            /** @var \Psr\Http\Message\ResponseInterface $ret */
            $ret = $this->client->$method($this->getUrl() . $url, $data);
            $response = $this->processResponse($ret);
            $response['response_code'] = $ret->getStatusCode();
        } catch (\GuzzleHttp\Exception\ClientException $e) {
            $this->log->error('Bad Response: ' . $e->getMessage());
            $this->log->error((string)$e->getRequest()->getBody());
            $response['response_status_code'] = $e->getCode();
            $response['response_status_message'] = $e->getMessage();
            $response = $this->processResponse($response);
            if ($e->hasResponse()) {
                $ret = $e->getResponse();
                $errorResponse = $e->getResponse();
                $this->log->error($errorResponse->getStatusCode() . ' ' . $errorResponse->getReasonPhrase());
                $body = $this->processResponse($errorResponse);
                $response = array_merge($response, $body);
            }
            if (!isset($response['response_code'])) {
                $response['response_code'] = $e->getCode();
            }
            $response['exception_code'] = $e->getCode();
        }
        return new Response($response);
    }

    /**
     * @return string
     */
    protected function getUrl()
    {
        $versionConfig = $this->configHelper->getVersionConfig($this->store);
        if ($this->configHelper->getApiConfigFlag('test_mode', $this->store)) {
            return $versionConfig->getTestdriveUrl();
        }
        return $versionConfig->getProductionUrl();
    }

    /**
     * @param mixed $response
     * @return array
     */
    protected function processResponse($response)
    {
        if (is_array($response)) {
            return $response;
        }
        try {
            $data = json_decode((string)$response->getBody(), true);
        } catch (\Exception $e) {
            $data = [
                'exception' => $e->getMessage()
            ];
        }
        if ($response->getStatusCode() === 401) {
            throw new KlarnaApiException(__($response->getReasonPhrase()));
        }
        $data['response_object'] = $response;
        $data['response_status_code'] = $response->getStatusCode();
        $data['response_status_message'] = $response->getReasonPhrase();
        return $data;
    }

    /**
     * @param string           $sessionId
     * @param RequestInterface $request
     * @return ResponseInterface
     */
    public function updateSession($sessionId, RequestInterface $request)
    {
        $response = $this->processRequest('/credit/' . self::API_VERSION . '/sessions/' . $sessionId, $request);
        if ($response->getResponseCode() === 204) {
            $data = $response->toArray();
            $data['session_id'] = $sessionId;
            return new Response($data);
        }
        return $response;
    }

    /**
     * @param string           $authorization_token
     * @param RequestInterface $request
     * @return ResponseInterface
     */
    public function placeOrder($authorization_token, RequestInterface $request)
    {
        return $this->processRequest('/credit/' . self::API_VERSION . '/authorizations/' . $authorization_token . '/order',
            $request);
    }

    /**
     * @param string $authorization_token
     * @return ResponseInterface
     */
    public function cancelOrder($authorization_token)
    {
        return $this->processRequest('/credit/' . self::API_VERSION . '/authorizations/' . $authorization_token, null,
            'delete');
    }
}
