<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */
namespace Klarna\Kp\Model;

use Klarna\Core\Helper\ConfigHelper;
use Klarna\Core\Exception as KlarnaException;
use Klarna\Kp\Api\Data\ResponseInterface;
use Klarna\Kp\Model\Payment\Kp;
use Magento\Checkout\Model\ConfigProviderInterface;

class KpConfigProvider implements ConfigProviderInterface
{
    /**
     * @var ConfigHelper
     */
    protected $config;

    /**
     * @var Session
     */
    protected $session;

    /**
     * Constructor
     *
     * @param ConfigHelper $config
     * @param Session      $checkoutSession
     */
    public function __construct(
        ConfigHelper $config,
        Session $checkoutSession
    ) {
        $this->config = $config;
        $this->session = $checkoutSession;
    }

    /**
     * Return payment config for frontend JS to use
     *
     * @return string[][]
     * @throws \Klarna\Core\Model\Api\Exception
     */
    public function getConfig()
    {
        $store = $this->session->getQuote()->getStore();
        $paymentConfig = [
            'payment' => [
                'klarna_kp' => [
                    'client_token'   => null,
                    'message'        => null,
                    'success'        => 0,
                    'debug'          => $this->config->getApiConfigFlag('debug', $store),
                    'prescreen'      => $this->config->getPaymentConfigFlag('pre_screen', $store),
                    'default_method' => $this->config->getPaymentConfigFlag('force_default', $store)
                ]
            ]
        ];

        if (!$this->config->getPaymentConfigFlag('active', $store, Kp::METHOD_CODE)) {
            $paymentConfig['payment']['klarna_kp']['message'] = __('Klarna Payments is not enabled');
            return $paymentConfig;
        }
        try {
            /** @var ResponseInterface $response */
            $response = $this->session->getApiResponse();
            if ($response === null) {
                $response = $this->session->init();
            }
            $klarnaQuote = $this->session->getKlarnaQuote();
            if ($klarnaQuote->getClientToken() === '') {
                $paymentConfig['payment']['klarna_kp']['message'] = __('Please check credentials');
                return $paymentConfig;
            }

            $paymentConfig['payment']['klarna_kp']['client_token'] = $klarnaQuote->getClientToken();
            $paymentConfig['payment']['klarna_kp']['authorization_token'] = $klarnaQuote->getAuthorizationToken();
            $paymentConfig['payment']['klarna_kp']['success'] = $response->getIsSuccessfull() ? 1 : 0;
            if (!$response->getIsSuccessfull()) {
                $paymentConfig['payment']['klarna_kp']['message'] = $response->getMessage();
            }
        } catch (KlarnaException $e) {
            $paymentConfig['payment']['klarna_kp']['message'] = $e->getMessage();
        }
        return $paymentConfig;
    }
}
