<?php
/**
 * This file is part of the Klarna KP module
 *
 * (c) Klarna Bank AB (publ)
 *
 * For the full copyright and license information, please view the NOTICE
 * and LICENSE files that were distributed with this source code.
 */
namespace Klarna\Kp\Model;

use Klarna\Core\Api\BuilderInterface;
use Klarna\Core\Model\Api\Exception as KlarnaApiException;
use Klarna\Kp\Api\CreditApiInterface;
use Klarna\Kp\Api\Data\RequestInterface;
use Klarna\Kp\Api\Data\ResponseInterface;
use Klarna\Kp\Api\QuoteInterface;
use Klarna\Kp\Api\QuoteRepositoryInterface;
use Klarna\Kp\Model\Api\Response;
use Klarna\Kp\Model\QuoteFactory as KlarnaQuoteFactory;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\App\Request\Http;
use Magento\Framework\App\State;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\PhpEnvironment\RemoteAddress;
use Magento\Framework\Session\Config\ConfigInterface;
use Magento\Framework\Session\SaveHandlerInterface;
use Magento\Framework\Session\SidResolverInterface;
use Magento\Framework\Session\StorageInterface;
use Magento\Framework\Session\ValidatorInterface;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Model\QuoteIdMaskFactory;
use Magento\Sales\Model\OrderFactory;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Class Session
 *
 * @package Klarna\Kp\Model
 */
class Session extends \Magento\Checkout\Model\Session
{
    /**
     * Request Builder
     *
     * @var BuilderInterface
     */
    protected $builder;

    /**
     * Klarna Payments API
     *
     * @var CreditApiInterface
     */
    protected $api;

    /**
     * Klarna Quote Repository
     *
     * @var QuoteRepositoryInterface
     */
    protected $klarnaQuoteRepository;

    /**
     * Klarna Quote Factory
     *
     * @var KlarnaQuoteFactory
     */
    protected $klarnaQuoteFactory;

    /**
     * @var ResponseInterface
     */
    protected $apiResponse;

    /**
     * @var QuoteInterface
     */
    protected $klarnaQuote;

    /**
     * Session constructor.
     *
     * @param Http                        $request
     * @param SidResolverInterface        $sidResolver
     * @param ConfigInterface             $sessionConfig
     * @param SaveHandlerInterface        $saveHandler
     * @param ValidatorInterface          $validator
     * @param StorageInterface            $storage
     * @param CookieManagerInterface      $cookieManager
     * @param CookieMetadataFactory       $cookieMetadataFactory
     * @param State                       $appState
     * @param OrderFactory                $orderFactory
     * @param CustomerSession             $customerSession
     * @param CartRepositoryInterface     $quoteRepository
     * @param RemoteAddress               $remoteAddress
     * @param ManagerInterface            $eventManager
     * @param StoreManagerInterface       $storeManager
     * @param CustomerRepositoryInterface $customerRepository
     * @param QuoteIdMaskFactory          $quoteIdMaskFactory
     * @param QuoteFactory                $quoteFactory
     * @param CreditApiInterface          $api
     * @param BuilderInterface            $builder
     * @param QuoteRepositoryInterface    $klarnaQuoteRepository
     * @param KlarnaQuoteFactory          $klarnaQuoteFactory
     */
    public function __construct(
        Http $request,
        SidResolverInterface $sidResolver,
        ConfigInterface $sessionConfig,
        SaveHandlerInterface $saveHandler,
        ValidatorInterface $validator,
        StorageInterface $storage,
        CookieManagerInterface $cookieManager,
        CookieMetadataFactory $cookieMetadataFactory,
        State $appState,
        OrderFactory $orderFactory,
        CustomerSession $customerSession,
        CartRepositoryInterface $quoteRepository,
        RemoteAddress $remoteAddress,
        ManagerInterface $eventManager,
        StoreManagerInterface $storeManager,
        CustomerRepositoryInterface $customerRepository,
        QuoteIdMaskFactory $quoteIdMaskFactory,
        QuoteFactory $quoteFactory,
        CreditApiInterface $api,
        BuilderInterface $builder,
        QuoteRepositoryInterface $klarnaQuoteRepository,
        KlarnaQuoteFactory $klarnaQuoteFactory
    ) {
        parent::__construct($request, $sidResolver, $sessionConfig, $saveHandler, $validator, $storage, $cookieManager,
            $cookieMetadataFactory, $appState, $orderFactory, $customerSession, $quoteRepository, $remoteAddress,
            $eventManager, $storeManager, $customerRepository, $quoteIdMaskFactory, $quoteFactory);
        $this->api = $api;
        $this->builder = $builder;
        $this->klarnaQuoteRepository = $klarnaQuoteRepository;
        $this->klarnaQuoteFactory = $klarnaQuoteFactory;
    }

    /**
     * Initialize Session
     *
     * @param string $sessionId
     * @return ResponseInterface
     * @throws \Klarna\Core\Model\Api\Exception
     */
    public function init($sessionId = null)
    {
        $klarnaResponse = $this->getApiResponse();
        if (!$klarnaResponse) {
            $klarnaResponse = $this->requestKlarnaSession($sessionId);

            $klarnaQuote = $this->generateKlarnaQuote($klarnaResponse);
            $this->setKlarnaQuote($klarnaQuote);
            $this->setApiResponse($klarnaResponse);
        }
        return $klarnaResponse;
    }

    /**
     * Start a Klarna Session
     *
     * @param string $sessionId
     * @return ResponseInterface
     * @throws \Klarna\Core\Model\Api\Exception
     */
    protected function requestKlarnaSession($sessionId = null)
    {
        if (empty($sessionId)) {
            return $this->initWithoutSession();
        }
        return $this->initWithSession($sessionId);
    }

    /**
     * Attempt to lookup existing session
     *
     * @param string $sessionId
     * @return ResponseInterface
     * @throws \Klarna\Core\Model\Api\Exception
     */
    protected function initWithSession($sessionId)
    {
        $data = $this->getGeneratedCreateRequest();
        $klarnaResponse = $this->updateOrCreateSession($sessionId, $data);

        if (!$klarnaResponse->getIsSuccessfull()) {
            throw new KlarnaApiException(__('Unable to initialize Klarna payments session'));
        }
        return $klarnaResponse;
    }

    /**
     * Get the create request
     *
     * @return RequestInterface|string[]
     */
    protected function getGeneratedCreateRequest()
    {
        return $this->builder->setObject($this->getQuote())->generateRequest(BuilderInterface::GENERATE_TYPE_CREATE)
                             ->getRequest();
    }

    /**
     * Update existing session. Create a new session if update fails
     *
     * @param string           $sessionId
     * @param RequestInterface $data
     * @return ResponseInterface
     */
    protected function updateOrCreateSession($sessionId, RequestInterface $data)
    {
        $resp = $this->api->updateSession($sessionId, $data);
        if ($resp->getIsSuccessfull()) {
            return $resp;
        }
        return $this->api->createSession($data);
    }

    /**
     * Set API Response
     *
     * @param ResponseInterface $klarnaQuote
     * @return $this
     */
    public function setApiResponse($klarnaQuote)
    {
        $this->apiResponse = $klarnaQuote;
        return $this;
    }

    /**
     * Create a new Klarna Session
     *
     * @return ResponseInterface
     * @throws \Klarna\Core\Model\Api\Exception
     */
    protected function initWithoutSession()
    {
        $data = $this->getGeneratedCreateRequest();
        $klarnaResponse = $this->initKlarnaQuote($data);

        if (!$klarnaResponse->getIsSuccessfull()) {
            throw new KlarnaApiException(__('Unable to initialize Klarna payments session'));
        }
        return $klarnaResponse;
    }

    /**
     * Initialize Klarna Quote
     *
     * @param RequestInterface $data
     * @return \Klarna\Kp\Api\Data\ResponseInterface
     */
    protected function initKlarnaQuote(RequestInterface $data)
    {
        try {
            $klarnaQuote = $this->klarnaQuoteRepository->getActiveByQuote($this->getQuote());
            $sessionId = $klarnaQuote->getSessionId();
            $resp = $this->updateOrCreateSession($sessionId, $data);
            if ($resp->getSessionId() !== $sessionId) {
                $this->klarnaQuoteRepository->markInactive($klarnaQuote);
            }
            return $resp;
        } catch (NoSuchEntityException $e) {
            return $this->api->createSession($data);
        }
    }

    /**
     * Lookup or create Klarna Quote
     *
     * @param ResponseInterface $klarnaResponse
     * @return QuoteInterface
     */
    protected function generateKlarnaQuote(ResponseInterface $klarnaResponse)
    {
        try {
            return $this->klarnaQuoteRepository->getBySessionId($klarnaResponse->getSessionId());
        } catch (NoSuchEntityException $e) {
            return $this->createNewQuote($klarnaResponse);
        }
    }

    /**
     * Create a new Klarna quote object
     *
     * @param ResponseInterface $resp
     * @return Quote
     */
    protected function createNewQuote(ResponseInterface $resp)
    {
        $quoteData = $resp->toArray();
        $quoteData['quote_id'] = $this->getQuote()->getId();
        $quoteData['is_active'] = 1;
        $klarnaQuote = $this->klarnaQuoteFactory->create(['data' => $quoteData]);
        $klarnaQuote->setDataChanges(true);
        $this->klarnaQuoteRepository->save($klarnaQuote);
        return $klarnaQuote;
    }

    /**
     * Set Klarna Quote
     *
     * @param QuoteInterface $klarnaQuote
     * @return $this
     */
    public function setKlarnaQuote(QuoteInterface $klarnaQuote)
    {
        $this->klarnaQuote = $klarnaQuote;
        return $this;
    }

    /**
     * Get Klarna Quote
     *
     * @return \Klarna\Kp\Api\QuoteInterface
     */
    public function getKlarnaQuote()
    {
        return $this->klarnaQuote;
    }

    /**
     * Get API Response
     *
     * @return ResponseInterface
     */
    public function getApiResponse()
    {
        return $this->apiResponse;
    }
}
