<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Adobe\Launch\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Encryption\EncryptorInterface;

/**
 * Configuration provider for Adobe Launch property provisioning.
 */
class ProvisioningConfigProvider
{
    private const LAUNCH_PROVISION_ADOBE_ORG_ID = 'launch_api_config/credentials/adobe_org_id';

    private const LAUNCH_PROVISION_CLIENT_ID = 'launch_api_config/credentials/client_id';

    private const LAUNCH_PROVISION_CLIENT_SECRET = 'launch_api_config/credentials/client_secret';

    private const LAUNCH_PROVISION_CLIENT_JWT = 'launch_api_config/credentials/jwt';

    private const LAUNCH_PROVISION_PROD_SUITE= 'launch_api_config/adobe_analytics/prod_suite';

    private const LAUNCH_PROVISION_STAGING_SUITE = 'launch_api_config/adobe_analytics/stage_suite';

    private const LAUNCH_PROVISION_DEV_SUITE = 'launch_api_config/adobe_analytics/dev_suite';

    private const LAUNCH_PROVISION_PROPERTY_NAME = 'launch_api_config/launch/property_name';

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param EncryptorInterface $encryptor
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        EncryptorInterface $encryptor
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->encryptor = $encryptor;
    }

    /**
     * Get Adobe Org ID from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getOrgID(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_ADOBE_ORG_ID, $scope);
    }

    /**
     * Get Client ID from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getClientID(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_CLIENT_ID, $scope);
    }

    /**
     * Get Client secret from configuration.
     *
     * @param string $scope
     * @return string
     */
    public function getClientSecret(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        $cypherText = $this->scopeConfig->getValue(self::LAUNCH_PROVISION_CLIENT_SECRET, $scope);
        return $this->encryptor->decrypt($cypherText);
    }

    /**
     * Get JWT from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getJWT(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_CLIENT_JWT, $scope);
    }

    /**
     * Get Production AA suite from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getProdSuite(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_PROD_SUITE, $scope);
    }

    /**
     * Get Stage AA suite from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getStageSuite(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_STAGING_SUITE, $scope);
    }

    /**
     * Get Dev AA suite from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getDevSuite(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_DEV_SUITE, $scope);
    }

    /**
     * Get Launch property name from configuration.
     *
     * @param string $scope
     * @return string|null
     */
    public function getPropertyName(?string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue(self::LAUNCH_PROVISION_PROPERTY_NAME, $scope);
    }
}
